package org.mule.munit.runner;

import static org.mule.metadata.java.api.JavaTypeLoader.JAVA;
import static org.mule.runtime.api.meta.Category.COMMUNITY;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.munit.common.model.*;
import org.mule.munit.runner.processors.ExcludeFlowSources;
import org.mule.runtime.api.message.Attributes;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ExecutionType;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.declaration.fluent.ConfigurableOutputDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ConfigurationDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ExtensionDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ParameterGroupDeclarer;
import org.mule.runtime.api.meta.model.display.LayoutModel;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.loader.ExtensionLoadingContext;
import org.mule.runtime.extension.api.loader.ExtensionLoadingDelegate;

public class MunitExtensionLoadingDelegate implements ExtensionLoadingDelegate {

  // General Attributes
  private static final String VERSION = "2.0.0-M1-SNAPSHOT";
  private static final String DESCRIPTION = "MUnit: Core components";
  private static final String NAME = "MUnit";
  private static final String VENDOR = "MuleSoft, Inc.";
  private static final MuleVersion MIN_MULE_VERSION = new MuleVersion("4.0.0-SNAPSHOT");
  // XML DSL Attributes
  private static final String PREFIX = "munit";
  private static final String NAMESPACE = "http://www.mulesoft.org/schema/mule/munit";
  private static final String SCHEMA_LOCATION = "http://www.mulesoft.org/schema/mule/munit/current/mule-munit.xsd";
  private static final String SCHEMA_VERSION = "2.0.0-SNAPSHOT";
  private static final String XSD_FILE_NAME = "mule-munit.xsd";
  // Configuration Attributes
  private static final String CONFIG_NAME = "config";
  private static final String CONFIG_DESCRIPTION = "MUnit configuration";
  private static final String EXCLUDE_FLOW_SOURCES_NAME = "exclude-flow-sources";
  private static final String EXCLUDE_FLOW_SOURCES_DESCRIPTION = "Flow sources to exclude";
  private static final int EXCLUDE_FLOW_SOURCES_ORDER = 2;
  // Operations - Test
  private static final String TEST_NAME = "test";
  private static final String TEST_DESCRIPTION = "The MUnit test flow";
  private static final String BEFORE_TEST_NAME = "before-test";
  private static final String BEFORE_TEST_DESCRIPTION = "Flow to be run before each MUnit test, one time per each test";
  private static final String AFTER_TEST_NAME = "after-test";
  private static final String AFTER_TEST_DESCRIPTION = "Flow to be run after each MUnit test, one time per each test";
  private static final String BEFORE_SUITE_NAME = "before-suite";
  private static final String BEFORE_SUITE_DESCRIPTION = "Flow to be run before any other in a MUnit suite, just one time";
  private static final String AFTER_SUITE_NAME = "after-suite";
  private static final String AFTER_SUITE_DESCRIPTION = "Flow to be run after any other in a MUnit suite, just one time";
  private static final String DESCRIPTION_NAME = "description";
  private static final String DESCRIPTION_DESCRIPTION = "Description message to be shown in case of test failure.";
  private static final String IGNORE_NAME = "ignore";
  private static final String IGNORE_DESCRIPTION = "Defines if the test must be ignored.";
  private static final String TAGS_NAME = "tags";
  private static final String TAGS_DESCRIPTION = "Defines the tags for the test.";
  private static final String EXPECTED_ERROR_TYPE_NAME = "expectedErrorType";
  private static final String EXPECTED_ERROR_TYPE_DESCRIPTION = "Error ID expected on this test.";
  private static final String EXPECTED_EXCEPTION_NAME = "expectedException";
  private static final String EXPECTED_EXCEPTION_DESCRIPTION = "Error expected on this test.";
  // Operations - Set Event
  private static final String SET_EVENT_NAME = "set-event";
  private static final String SET_EVENT_DESCRIPTION = "Defines the event to be used for testing";
  private static final String CLONE_ORIGINAL_EVENT_NAME = "cloneOriginalEvent";
  private static final String CLONE_ORIGINAL_EVENT_DESCRIPTION = "Define if the original Event should be cloned";
  private static final String PAYLOAD_NAME = "payload";
  private static final String PAYLOAD_DESCRIPTION = "The payload to be set.";
  private static final String ATTRIBUTES_NAME = "attributes";
  private static final String ATTRIBUTES_DESCRIPTION = "Attributes to be set.";
  private static final String ERROR_NAME = "error";
  private static final String ERROR_DESCRIPTION = "Error to be set.";
  private static final String VARIABLE_NAME = "variables";
  private static final String VARIABLE_DESCRIPTION = "Variables to be set.";
  private static final String INBOUND_PROPERTIES_NAME = "inbound-properties";
  private static final String INBOUND_PROPERTIES_DESCRIPTION = "Inbound properties to be set.";
  private static final String OUTBOUND_PROPERTIES_NAME = "outbound-properties";
  private static final String OUTBOUND_PROPERTIES_DESCRIPTION = "Outbound properties to be set.";
  private static final String INBOUND_ATTACHMENTS_NAME = "inbound-attachments";
  private static final String INBOUND_ATTACHMENTS_DESCRIPTION = "Inbound attachments to be set.";
  private static final String OUTBOUND_ATTACHMENTS_NAME = "outbound-attachments";
  private static final String OUTBOUND_ATTACHMENTS_DESCRIPTION = "Outbound attachments to be set.";
  // Operations - Set Null Payload
  private static final String SET_NULL_PAYLOAD_NAME = "set-null-payload";
  private static final String SET_NULL_PAYLOAD_DESCRIPTION = "Defines a Null payload for testing";

  private ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader();
  private BaseTypeBuilder typeBuilder = BaseTypeBuilder.create(JAVA);

  @Override
  public void accept(ExtensionDeclarer extensionDeclarer, ExtensionLoadingContext extensionLoadingContext) {

    declareGeneral(extensionDeclarer);
    declareXmlDsl(extensionDeclarer);
    declareConfiguration(extensionDeclarer);
    declareOperations(extensionDeclarer);
  }

  private void declareGeneral(ExtensionDeclarer extensionDeclarer) {
    extensionDeclarer
        .named(NAME)
        .describedAs(DESCRIPTION)
        .onVersion(VERSION)
        .fromVendor(VENDOR)
        .withCategory(COMMUNITY)
        .withMinMuleVersion(MIN_MULE_VERSION);
  }

  private void declareXmlDsl(ExtensionDeclarer extensionDeclarer) {
    XmlDslModel xmlDslModel = XmlDslModel.builder()
        .setPrefix(PREFIX)
        .setNamespace(NAMESPACE)
        .setSchemaLocation(SCHEMA_LOCATION)
        .setSchemaVersion(SCHEMA_VERSION)
        .setXsdFileName(XSD_FILE_NAME).build();
    extensionDeclarer.withXmlDsl(xmlDslModel);
  }

  private void declareConfiguration(ExtensionDeclarer extensionDeclarer) {
    ConfigurationDeclarer configurationDeclarer = extensionDeclarer
        .withConfig(CONFIG_NAME)
        .describedAs(CONFIG_DESCRIPTION);
    ParameterGroupDeclarer parameterGroupDeclarer = configurationDeclarer.onDefaultParameterGroup();
    parameterGroupDeclarer
        .withOptionalParameter(EXCLUDE_FLOW_SOURCES_NAME)
        .describedAs(EXCLUDE_FLOW_SOURCES_DESCRIPTION)
        .ofType(typeLoader.load(ExcludeFlowSources.class))
        .withExpressionSupport(ExpressionSupport.NOT_SUPPORTED)
        .withLayout(LayoutModel.builder().order(EXCLUDE_FLOW_SOURCES_ORDER).build());
  }

  private void declareOperations(ExtensionDeclarer extensionDeclarer) {
    declareMunitTestFlow(extensionDeclarer);
    declareMunitFlow(extensionDeclarer, BEFORE_TEST_NAME, BEFORE_TEST_DESCRIPTION);
    declareMunitFlow(extensionDeclarer, AFTER_TEST_NAME, AFTER_TEST_DESCRIPTION);
    declareMunitFlow(extensionDeclarer, BEFORE_SUITE_NAME, BEFORE_SUITE_DESCRIPTION);
    declareMunitFlow(extensionDeclarer, AFTER_SUITE_NAME, AFTER_SUITE_DESCRIPTION);
    declareSetEvent(extensionDeclarer);
    declareSetNullPayload(extensionDeclarer);
  }

  private void declareMunitTestFlow(ExtensionDeclarer extensionDeclarer) {
    ConfigurableOutputDeclarer configurableOutputDeclarer = declareMunitFlow(extensionDeclarer, TEST_NAME, TEST_DESCRIPTION);
    ParameterGroupDeclarer parameterGroupDeclarer = configurableOutputDeclarer.onDefaultParameterGroup();
    addOptionalParameter(IGNORE_NAME, IGNORE_DESCRIPTION, Boolean.class, parameterGroupDeclarer);
    addOptionalParameter(TAGS_NAME, TAGS_DESCRIPTION, String.class, parameterGroupDeclarer);
    addOptionalParameter(EXPECTED_ERROR_TYPE_NAME, EXPECTED_ERROR_TYPE_DESCRIPTION, String.class, parameterGroupDeclarer);
    addOptionalParameter(EXPECTED_EXCEPTION_NAME, EXPECTED_EXCEPTION_DESCRIPTION, String.class, parameterGroupDeclarer);

  }

  private ConfigurableOutputDeclarer declareMunitFlow(ExtensionDeclarer extensionDeclarer, String name, String description) {
    ConfigurableOutputDeclarer configurableOutputDeclarer = extensionDeclarer
        .withScope(name)
        .describedAs(description)
        .withExecutionType(ExecutionType.BLOCKING);
    configurableOutputDeclarer.withOutputAttributes().ofType(typeLoader.load(Attributes.class));
    configurableOutputDeclarer.withOutput().ofType(typeBuilder.anyType().build());
    addOptionalParameter(DESCRIPTION_NAME, DESCRIPTION_DESCRIPTION, String.class,
                         configurableOutputDeclarer.onDefaultParameterGroup());
    return configurableOutputDeclarer;
  }

  private void declareSetEvent(ExtensionDeclarer extensionDeclarer) {
    ConfigurableOutputDeclarer configurableOutputDeclarer = extensionDeclarer
        .withOperation(SET_EVENT_NAME)
        .describedAs(SET_EVENT_DESCRIPTION)
        .withExecutionType(ExecutionType.CPU_LITE);
    configurableOutputDeclarer.withOutputAttributes().ofType(typeLoader.load(Attributes.class));
    configurableOutputDeclarer.withOutput().ofType(typeBuilder.anyType().build());

    ParameterGroupDeclarer declarer = configurableOutputDeclarer.onDefaultParameterGroup();
    addOptionalParameter(CLONE_ORIGINAL_EVENT_NAME, CLONE_ORIGINAL_EVENT_DESCRIPTION, Boolean.class, declarer);
    addOptionalParameter(PAYLOAD_NAME, PAYLOAD_DESCRIPTION, Payload.class, declarer);
    addOptionalParameter(ATTRIBUTES_NAME, ATTRIBUTES_DESCRIPTION, EventAttributes.class, declarer);
    addOptionalParameter(ERROR_NAME, ERROR_DESCRIPTION, UntypedEventError.class, declarer);
    addOptionalListParameter(VARIABLE_NAME, VARIABLE_DESCRIPTION, Variable.class, declarer);
    addOptionalListParameter(INBOUND_PROPERTIES_NAME, INBOUND_PROPERTIES_DESCRIPTION, Property.class, declarer);
    addOptionalListParameter(OUTBOUND_PROPERTIES_NAME, OUTBOUND_PROPERTIES_DESCRIPTION, Property.class, declarer);
    addOptionalListParameter(INBOUND_ATTACHMENTS_NAME, INBOUND_ATTACHMENTS_DESCRIPTION, Attachment.class, declarer);
    addOptionalListParameter(OUTBOUND_ATTACHMENTS_NAME, OUTBOUND_ATTACHMENTS_DESCRIPTION, Attachment.class, declarer);
  }

  private void declareSetNullPayload(ExtensionDeclarer extensionDeclarer) {
    ConfigurableOutputDeclarer configurableOutputDeclarer = extensionDeclarer
        .withOperation(SET_NULL_PAYLOAD_NAME)
        .describedAs(SET_NULL_PAYLOAD_DESCRIPTION)
        .withExecutionType(ExecutionType.CPU_LITE);
    configurableOutputDeclarer.withOutputAttributes().ofType(typeLoader.load(Attributes.class));
    configurableOutputDeclarer.withOutput().ofType(typeBuilder.anyType().build());
  }

  private void addOptionalParameter(String name, String description, Class<?> clazz, ParameterGroupDeclarer declarer) {
    addOptionalParameter(name, description, typeLoader.load(clazz), declarer);
  }

  private void addOptionalListParameter(String name, String description, Class<?> clazz, ParameterGroupDeclarer declarer) {
    addOptionalParameter(name, description, typeBuilder.arrayType().of(typeLoader.load(clazz)).build(), declarer);
  }

  private void addOptionalParameter(String name, String description, MetadataType type, ParameterGroupDeclarer declarer) {
    declarer
        .withOptionalParameter(name)
        .describedAs(description)
        .ofType(type);
  }

}
