/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.munit.common.protocol.message.TestStatus.*;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.munit.common.event.EventBuilder;
import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.common.protocol.message.TestStatus;
import org.mule.munit.runner.output.DefaultOutputHandler;
import org.mule.munit.runner.output.TestOutputHandler;
import org.mule.munit.runner.processors.MunitFlow;
import org.mule.munit.runner.remote.api.notifiers.DummySuiteRunEventListener;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.exception.MessagingException;

/**
 * MUnit Suite. This class is part of the execution model of MUnit.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class Suite {

  private transient Log logger = LogFactory.getLog(this.getClass());

  private String name;
  private List<MunitFlow> beforeSuiteFlows;
  private List<Test> tests;
  private List<MunitFlow> afterSuiteFlows;

  private TestOutputHandler outputHandler;
  private SuiteRunEventListener suiteRunEventListener;

  public Suite(String name) {
    this.name = name;
    this.beforeSuiteFlows = new ArrayList<>();
    this.tests = new ArrayList<>();
    this.afterSuiteFlows = new ArrayList<>();

    this.outputHandler = new DefaultOutputHandler();
    this.suiteRunEventListener = new DummySuiteRunEventListener();
  }

  public void setOutputHandler(TestOutputHandler outputHandler) {
    checkNotNull(outputHandler, "The outputHandler must not be null");
    this.outputHandler = outputHandler;
  }

  public void setSuiteRunEventListener(SuiteRunEventListener suiteRunnerEventListener) {
    checkNotNull(suiteRunEventListener, "The suiteRunEventListener must not be null");
    this.suiteRunEventListener = suiteRunnerEventListener;
  }

  public String getName() {
    return name;
  }

  public void addBeforeSuite(MunitFlow beforeSuite) {
    beforeSuiteFlows.add(beforeSuite);
  }

  public void addTest(Test test) {
    tests.add(test);
  }

  public void addAfterSuite(MunitFlow afterSuite) {
    afterSuiteFlows.add(afterSuite);
  }

  public int getNumberOfTests() {
    return tests.size();
  }

  public boolean allTestsIgnored() {
    return tests.stream().allMatch(Test::isIgnore);
  }

  /**
   * Runs all the tests of the suite
   *
   * @return The Result of the suite execution
   * @throws Exception If the suite failed for one reason.
   */
  public SuiteResult run() throws Exception {
    SuiteResult result = new SuiteResult(name);
    try {
      runBeforeSuites();

      for (Test test : tests) {
        suiteRunEventListener.notifyTestStart(test.getName());

        TestResult testResult = test.run();
        result.add(testResult);

        suiteRunEventListener.notifyTestEnd(testResult.getName(), testResult.getStackTrace(), testResult.getStatus(),
                                            testResult.getElapsedTime());
      }
    } catch (Throwable e) {
      logger.error("Could not Run the suite: " + name, e);
    } finally {
      runAfterSuites();
    }
    return result;
  }

  private void runBeforeSuites() throws Throwable {
    String flowName = StringUtils.EMPTY;
    try {
      logger.debug("Executing Before Suite scopes...");
      for (MunitFlow beforeSuiteFlow : beforeSuiteFlows) {
        flowName = beforeSuiteFlow.getName();
        Event event = EventBuilder.buildGenericMuleEvent(beforeSuiteFlow);
        suiteRunEventListener.notifyBeforeSuiteStart(flowName);
        process(beforeSuiteFlow, event);
        suiteRunEventListener.notifyBeforeSuiteEnd(flowName, StringUtils.EMPTY, SUCCESS);
      }
    } catch (MessagingException e) {
      Throwable cause = e.getRootCause();
      handleBeforeSuiteEndByException(flowName, cause);
      throw cause;
    } catch (Throwable e) {
      handleBeforeSuiteEndByException(flowName, e);
      throw e;
    }
  }

  private void runAfterSuites() {
    String flowName = StringUtils.EMPTY;
    try {
      logger.debug("Executing After Suite scopes...");
      for (MunitFlow afterSuiteFlow : afterSuiteFlows) {
        flowName = afterSuiteFlow.getName();
        Event event = EventBuilder.buildGenericMuleEvent(afterSuiteFlow);
        suiteRunEventListener.notifyAfterSuiteStart(flowName);
        process(afterSuiteFlow, event);
        suiteRunEventListener.notifyAfterSuiteEnd(flowName, StringUtils.EMPTY, SUCCESS);
      }
    } catch (MessagingException e) {
      handleAfterSuiteEndByException(flowName, e.getRootCause());
    } catch (Throwable e) {
      handleAfterSuiteEndByException(flowName, e);
    }
  }

  private void handleBeforeSuiteEndByException(String flowName, Throwable cause) {
    String message = String.format("Before suite %s execution failed", flowName);
    logger.error(message, cause);
    String stackTrace = getStackTrace(cause, message);
    suiteRunEventListener.notifyBeforeSuiteEnd(flowName, stackTrace, getStatus(cause));
  }

  private void handleAfterSuiteEndByException(String flowName, Throwable cause) {
    String message = String.format("After suite %s execution failed", flowName);
    logger.error(message, cause);
    String stackTrace = getStackTrace(cause, message);
    suiteRunEventListener.notifyAfterSuiteEnd(flowName, stackTrace, getStatus(cause));
  }

  private void process(MunitFlow munitFlow, Event event) throws MuleException {
    outputHandler.printDescription(munitFlow.getName(), munitFlow.getDescription());
    munitFlow.process(event);
  }

  private String getStackTrace(Throwable cause, String message) {
    RuntimeException e = new RuntimeException(message, cause);
    return ExceptionUtils.getStackTrace(e);
  }

  private TestStatus getStatus(Throwable e) {
    return e instanceof AssertionError ? FAILURE : ERROR;
  }

}
