/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model;

import java.util.List;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.munit.common.event.EventBuilder;
import org.mule.munit.runner.output.TestOutputHandler;
import org.mule.munit.runner.processors.MunitFlow;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.munit.runner.processors.MunitTestFlow;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.api.construct.FlowConstruct;
import org.mule.runtime.core.exception.MessagingException;

/**
 * MUnit Test. This class is part of the execution model of MUnit.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class Test {

  private transient Log logger = LogFactory.getLog(this.getClass());

  /**
   * The MUnit flows that have to be run before the MUnit test.
   */
  private List<MunitFlow> before;

  /**
   * The MUnit flows that have to be run after the MUnit test.
   */
  private List<MunitFlow> after;

  /**
   * The MUnit test.
   */
  private MunitTestFlow test;

  /**
   * The Output handler to be use.
   */
  private TestOutputHandler outputHandler;

  private MunitModule munitModule;

  public Test(List<MunitFlow> before, MunitTestFlow test, List<MunitFlow> after, TestOutputHandler outputHandler,
              MunitModule munitModule) {
    this.before = before;
    this.test = test;
    this.after = after;
    this.outputHandler = outputHandler;
    this.munitModule = munitModule;
  }

  public String getName() {
    return test.getName();
  }

  public boolean isIgnore() {
    return test.isIgnore();
  }

  public TestResult run() {
    TestResult result = new TestResult(getName());
    logger.debug("About to run MUnit test: " + getName());

    if (test.isIgnore()) {
      logger.debug("MUnit test: " + getName() + " is ignored it won't run.");
      result.setSkipped();
      result.setElapsedTime(0);
      return result;
    }

    long testStartTime = System.currentTimeMillis();
    Event event = muleEvent(test);

    try {
      event = runBefore(event);

      showDescription();

      event = test.run(event);
    } catch (AssertionError t) {
      result.setFailure(ExceptionUtils.getStackTrace(t));
    } catch (Throwable e) {
      result.setError(ExceptionUtils.getStackTrace(e));
      if (e instanceof MessagingException) {
        event = ((MessagingException) e).getEvent();
      }
    } finally {
      munitModule.reset();
      runAfter(result, event);
    }

    result.setElapsedTime(System.currentTimeMillis() - testStartTime);
    return result;
  }

  private Event runBefore(Event event) throws Throwable {
    logger.debug("Running before test scopes...");
    try {
      return run(event, before);
    } catch (MessagingException e) {
      throw handleRunBeforeFailure(e, e.getRootCause());
    } catch (MuleException e) {
      throw handleRunBeforeFailure(e, e.getCause());
    }
  }

  private void runAfter(TestResult result, Event event) {
    logger.debug("Running after test scopes...");
    try {
      run(event, after);
    } catch (MessagingException e) {
      handleRunAfterFailure(result, e, e.getRootCause());
    } catch (MuleException e) {
      handleRunAfterFailure(result, e, e.getCause());
    }
  }

  private void handleRunAfterFailure(TestResult result, MuleException e, Throwable cause) {
    if (null != cause && cause.getClass().isAssignableFrom(AssertionError.class)) {
      result.setFailure(ExceptionUtils.getStackTrace(cause));
    } else {
      result.setError(ExceptionUtils.getStackTrace(e));
    }
  }

  private Throwable handleRunBeforeFailure(MuleException e, Throwable cause) throws Throwable {
    if (null != cause && cause.getClass().isAssignableFrom(AssertionError.class)) {
      return cause;
    } else {
      return e;
    }
  }

  private Event run(Event event, List<MunitFlow> flows) throws MuleException {
    if (flows != null) {
      for (MunitFlow flow : flows) {
        outputHandler.printDescription(flow.getName(), flow.getDescription());
        event = flow.process(event);
      }
    }
    return event;
  }

  private void showDescription() {
    outputHandler.printDescription(test.getName(), test.getDescription());
  }

  protected Event muleEvent(FlowConstruct flow) {
    return new EventBuilder(flow).withPayload("").build(); // TODO REMOVE THIS
  }

}
