/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model.builders;


import org.apache.commons.lang.StringUtils;
import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.runner.model.Suite;
import org.mule.munit.runner.model.Test;
import org.mule.munit.runner.output.DefaultOutputHandler;
import org.mule.munit.runner.output.TestOutputHandler;
import org.mule.munit.runner.processors.*;
import org.mule.munit.runner.remote.api.notifiers.DummySuiteRunEventListener;

import java.util.*;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.munit.runner.processors.MunitTestFlow.NO_TAG_TOKEN;

/**
 * Creates a MUnit Suite and its Tests
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SuiteBuilder {

  protected String suiteName;
  protected MunitModule munitModule;

  protected Set<String> tags;
  protected List<String> testNames;
  protected TestOutputHandler outputHandler;
  protected SuiteRunEventListener suiteRunEventListener;

  public SuiteBuilder(String suiteName, MunitModule munitModule) {
    checkArgument(StringUtils.isNotBlank(suiteName), "The suiteName must not be null nor empty");
    checkNotNull(munitModule, "The munitModulemust not be null");

    this.suiteName = suiteName;
    this.munitModule = munitModule;
    this.outputHandler = new DefaultOutputHandler();
    this.suiteRunEventListener = new DummySuiteRunEventListener();
  }

  public SuiteBuilder withTestNames(List<String> testNames) {
    this.testNames = testNames;
    return this;
  }

  public SuiteBuilder withTags(Set<String> tags) {
    this.tags = tags;
    return this;
  }

  public SuiteBuilder withOutputHandler(TestOutputHandler outputHandler) {
    checkNotNull(outputHandler, "The outputHandler must not be null");
    this.outputHandler = outputHandler;
    return this;
  }

  public SuiteBuilder withSuiteRunnerEventListener(SuiteRunEventListener suiteRunEventListener) {
    checkNotNull(suiteRunEventListener, "The suiteRunEventListener must not be null");
    this.suiteRunEventListener = suiteRunEventListener;
    return this;
  }

  /**
   * Builds the Suite with a particular suite name, based on the mule context
   * 
   * @return The Suite Object
   */
  public Suite build() {
    List<MunitFlow> before = lookupFlows(MunitBeforeTest.class);
    List<MunitFlow> after = lookupFlows(MunitAfterTest.class);
    Collection<MunitTestFlow> testFlows = lookupTests();

    Suite suite = new Suite(suiteName);
    suite.setOutputHandler(outputHandler);
    suite.setSuiteRunEventListener(suiteRunEventListener);

    beforeSuites().forEach(suite::addBeforeSuite);
    createTests(before, after, testFlows).forEach(suite::addTest);
    afterSuites().forEach(suite::addAfterSuite);

    return suite;
  }

  private List<Test> createTests(List<MunitFlow> before, List<MunitFlow> after, Collection<MunitTestFlow> testFlows) {
    List<Test> tests = new ArrayList<>();
    tests.addAll(testFlows.stream()
        .filter(testFlow -> shouldRunTest(testFlow, testNames, tags))
        .map(testFlow -> test(before, testFlow, after))
        .collect(Collectors.toList()));

    return tests;
  }

  /**
   * Create a test
   *
   * @param beforeTest Munit flows to be run before the test
   * @param test Munit Flow that represents the test
   * @param afterTest Munit flows to be run after the test
   * @return The Test Object
   */
  protected Test test(List<MunitFlow> beforeTest, MunitTestFlow test, List<MunitFlow> afterTest) {
    return new Test(beforeTest, test, afterTest, outputHandler, munitModule);
  }

  private boolean shouldRunTest(MunitTestFlow munitTestFlow, List<String> testNameList, Set<String> tags) {
    return isTestInList(munitTestFlow, testNameList) && isTagInTest(munitTestFlow, tags);
  }

  private boolean isTestInList(MunitTestFlow munitTestFlow, List<String> testNameList) {
    if (null == testNameList || testNameList.isEmpty()) {
      return true;
    }
    return testNameList.stream().filter(StringUtils::isNotBlank).anyMatch(name -> munitTestFlow.getName().matches(name));
  }

  private boolean isTagInTest(MunitTestFlow munitTestFlow, Set<String> tags) {
    return tags == null || tags.isEmpty() || shouldRunWithNoTag(munitTestFlow, tags)
        || tagsInCommon(munitTestFlow.getTags(), tags);
  }

  private boolean tagsInCommon(Set<String> testTags, Set<String> userTags) {
    return !Collections.disjoint(toLowerCaseAndTrim(testTags), toLowerCaseAndTrim(userTags));
  }

  private Set<String> toLowerCaseAndTrim(Set<String> tags) {
    return tags.stream().map(StringUtils::lowerCase).map(String::trim).collect(Collectors.toSet());
  }

  private boolean shouldRunWithNoTag(MunitTestFlow munitTestFlow, Set<String> tags) {
    return munitTestFlow.getTags().isEmpty() && tags.stream().anyMatch(tag -> tag.trim().equalsIgnoreCase(NO_TAG_TOKEN));
  }

  private List<MunitFlow> beforeSuites() {
    return lookupFlows(MunitBeforeSuite.class);
  }

  private List<MunitFlow> afterSuites() {
    return lookupFlows(MunitAfterSuite.class);
  }

  private List<MunitFlow> lookupFlows(Class<? extends MunitFlow> munitClass) {
    return new ArrayList<>(munitModule.getMuleContext().getRegistry().lookupObjects(munitClass));
  }

  private Collection<MunitTestFlow> lookupTests() {
    return new ArrayList<>(munitModule.getMuleContext().getRegistry().lookupObjects(MunitTestFlow.class));
  }
}
