/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.processors;

import org.apache.commons.lang.StringUtils;
import org.mule.munit.common.util.MunitExpressionWrapper;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.construct.FlowConstruct;
import org.mule.runtime.core.api.construct.FlowConstructAware;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.config.i18n.CoreMessages;
import org.mule.runtime.core.exception.MessagingException;

import javax.inject.Inject;

/**
 * Generic Munit Processor.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class MunitProcessor implements Initialisable, Processor, FlowConstructAware {

  @Inject
  protected MuleContext muleContext;

  protected FlowConstruct flowConstruct;
  protected MunitExpressionWrapper expressionWrapper;

  public void initialise() throws InitialisationException {
    expressionWrapper = new MunitExpressionWrapper(muleContext.getExpressionManager());
  }

  /**
   * Executes the message processor code. In case of an assertion error it throws a new exception with the mule stack trace
   * 
   * @param event The mule event to be processed.
   * @return The result mule event
   * @throws MuleException In case of error. If the assertion fails, it throws an {@link AssertionError}
   * 
   */
  public Event process(Event event) throws MuleException {
    try {
      return doProcess(event);
    } catch (AssertionError error) {
      AssertionError exception = new AssertionError(getMessage(error));
      throw exception;
    } catch (Exception e) {
      throw new MessagingException(CoreMessages.failedToInvoke(getProcessor()), event, e);
    }
  }

  /**
   * The method that do the actual process
   *
   * @param event * The mule Event
   * @return the processed event
   */
  // TODO do we need a module??
  protected abstract Event doProcess(Event event);

  /**
   * @return The name of the processor
   * 
   */
  // TODO do we need a this??
  protected abstract String getProcessor();

  public void setMuleContext(MuleContext context) {
    this.muleContext = context;
  }

  @Override
  public void setFlowConstruct(FlowConstruct flowConstruct) {
    this.flowConstruct = flowConstruct;
  }

  private String getMessage(AssertionError error) {
    String message = error.getMessage();
    if (StringUtils.isEmpty(message)) {
      return this.getProcessor();
    }
    return message;
  }
}
