/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote.api.server;

import com.google.gson.Gson;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.common.protocol.message.RunMessage;
import org.mule.munit.runner.SuiteRunner;
import org.mule.munit.runner.model.Suite;
import org.mule.munit.runner.model.builders.SuiteBuilder;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.munit.runner.remote.api.notifiers.ObjectOutputStreamNotifier;
import org.mule.munit.runner.remote.api.notifiers.StreamNotifier;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.Socket;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static org.mule.munit.common.protocol.message.MessageField.*;
import static org.mule.munit.common.protocol.message.MessageID.RUN_SUITE;

public class RunMessageHandler {

  private transient Log log = LogFactory.getLog(this.getClass());

  private ObjectInputStream in;
  private ObjectOutputStream out;
  private MunitModule munitModule;

  public RunMessageHandler(Socket socket, MunitModule munitModule) throws IOException {
    this.munitModule = munitModule;

    try {
      this.out = new ObjectOutputStream(socket.getOutputStream());
      this.in = new ObjectInputStream(socket.getInputStream());
    } catch (IOException e) {
      log.error("Message received but we fail to open streams");
      throw e;
    }
  }

  public void handle() throws IOException, ClassNotFoundException {
    String message = (String) in.readObject();
    RunMessage runMessage = new Gson().fromJson(message, RunMessage.class);

    if (RUN_SUITE == runMessage.getId()) {
      runSuite(runMessage);
    }
  }

  private void runSuite(RunMessage runMessage) {
    SuiteRunEventListener listener = buildListenerContainer(runMessage.get(RUN_TOKEN_KEY));
    try {
      SuiteRunner runner = new SuiteRunner(buildSuite(runMessage, listener), listener);
      runner.run();
    } catch (Throwable e) {
      e.printStackTrace();
      listener.notifySuiteUnexpectedError(ExceptionUtils.getStackTrace(e));
    }
  }

  protected Suite buildSuite(RunMessage runMessage, SuiteRunEventListener listener) {
    SuiteBuilder suiteBuilder = new SuiteBuilder(getSuiteName(runMessage), munitModule)
        .withTestNames(getTestNames(runMessage))
        .withTags(getTags(runMessage))
        .withSuiteRunnerEventListener(listener);

    Suite suite = suiteBuilder.build();
    return suite;
  }

  private String getSuiteName(RunMessage runMessage) {
    return runMessage.get(MUNIT_SUITE_KEY);
  }

  private SuiteRunEventListener buildListenerContainer(String runToken) {
    RunEventListenerContainer listenerDecorator = new RunEventListenerContainer();
    listenerDecorator.addNotificationListener(new ObjectOutputStreamNotifier(runToken, out));
    listenerDecorator.addNotificationListener(new StreamNotifier(System.out));
    return listenerDecorator;
  }

  private List<String> getTestNames(RunMessage runMessage) {
    String testNames = runMessage.get(TEST_NAMES_KEY);
    if (StringUtils.isBlank(testNames)) {
      return Collections.emptyList();
    }
    return newArrayList(testNames.split(SEPARATOR_TOKEN));
  }

  private Set<String> getTags(RunMessage runMessage) {
    String tags = runMessage.get(TAGS_KEY);
    if (StringUtils.isBlank(tags)) {
      return Collections.emptySet();
    }
    return newHashSet(tags.split(SEPARATOR_TOKEN));
  }
}
