/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote.api.server;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.api.meta.AnnotatedObject;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.locator.ConfigurationComponentLocator;

import javax.inject.Inject;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * The goal of this class is to offer an entry point to the container through the munit-runner plugin with the objective to run
 * MUnit Test Suites
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerServer implements Runnable, Lifecycle {

  private static final int SOCKET_TIMEOUT_MILLIS = 5000;
  private static final String MUNIT_SERVER_PORT = "munit.server.port";

  private transient Log log = LogFactory.getLog(this.getClass());

  @Inject
  private MuleContext muleContext;

  @Inject
  ConfigurationComponentLocator configurationComponentLocator;

  private int port;
  private boolean keepRunning = true;

  public RunnerServer() {}

  public void run() {
    ServerSocket providerSocket = null;
    try {
      port = Integer.parseInt(System.getProperty(MUNIT_SERVER_PORT));
      providerSocket = new ServerSocket(port, 10);
      providerSocket.setSoTimeout(SOCKET_TIMEOUT_MILLIS);
      log.debug("MUnit server started listening in port [" + port + "]...");

      do {
        log.info("Waiting for client connection ");
        Socket connection = providerSocket.accept();
        log.info("Client connection received from " + connection.getInetAddress().getHostName() + " - " + keepRunning);

        RunMessageHandler commander = new RunMessageHandler(connection, getMunitModule());
        commander.handle();
      } while (keepRunning);

    } catch (SocketTimeoutException timeoutException) {
      log.debug("MUnit server time out");
      if (keepRunning) {
        log.error("Client connection timeout after " + String.valueOf(SOCKET_TIMEOUT_MILLIS) + " milliseconds");
      }
    } catch (IOException ioException) {
      log.error("Failed to start MUnit server in port " + port);
    } catch (ClassNotFoundException e) {
      log.error("Fail to deserialize message.", e);
    } finally {
      try {
        log.debug("Shutting down MUnit server running in port [" + port + "]...");

        if (null != providerSocket) {
          providerSocket.close();
        }
        keepRunning = false;
        log.debug("MUnit server shutdown");
      } catch (IOException ioException) {
        log.debug("MUnit server error during shutdown.");
      }
    }
  }

  private MunitModule getMunitModule() {
    List<AnnotatedObject> annotatedObjects =
        configurationComponentLocator.find(ComponentIdentifier.buildFromStringRepresentation("munit:config"));
    if (!annotatedObjects.isEmpty()) {
      return (MunitModule) annotatedObjects.get(0);
    }
    throw new IllegalStateException("Missing element [munit:config]");
  }

  public void setMuleContext(MuleContext context) {
    this.muleContext = context;
  }

  @Override
  public void initialise() throws InitialisationException {
    log.debug("Initializing MUnit server...");
    muleContext.getSchedulerService().ioScheduler().schedule(this, 0, TimeUnit.SECONDS);
  }

  @Override
  public void start() throws MuleException {

  }

  @Override
  public void stop() throws MuleException {
    log.debug("Stop signal received, shooting down MUnit server...");
    keepRunning = false;
  }

  @Override
  public void dispose() {
    log.debug("Dispose signal received, shooting down MUnit server...");
    keepRunning = false;
  }

}
