/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.config;

import static org.mule.munit.runner.config.MunitXmlNamespaceInfoProvider.NAME_SPACE;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromChildCollectionConfiguration;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromChildConfiguration;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromSimpleParameter;
import static org.mule.runtime.dsl.api.component.TypeDefinition.fromType;

import org.mule.munit.common.model.Attachment;
import org.mule.munit.common.model.EventAttributes;
import org.mule.munit.common.model.Payload;
import org.mule.munit.common.model.Property;
import org.mule.munit.common.model.UntypedEventError;
import org.mule.munit.common.model.Variable;
import org.mule.munit.runner.flow.AfterSuite;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeSuite;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.flow.TestFlow;
import org.mule.munit.runner.processors.ExcludeFlowSources;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.munit.runner.processors.SetEventProcessor;
import org.mule.runtime.api.meta.AnnotatedObject;
import org.mule.runtime.core.privileged.processor.objectfactory.MessageProcessorChainObjectFactory;
import org.mule.runtime.dsl.api.component.ComponentBuildingDefinition;
import org.mule.runtime.dsl.api.component.ComponentBuildingDefinitionProvider;

import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * Provides the Component Definition for the MUnit schema
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitComponentBuildingDefinitionProvider implements ComponentBuildingDefinitionProvider {

  public static final String CONFIG = "config";
  public static final String TEST = "test";
  public static final String BEFORE_TEST = "before-test";
  public static final String AFTER_TEST = "after-test";
  public static final String BEFORE_SUITE = "before-suite";
  public static final String AFTER_SUITE = "after-suite";

  private static final String SET_EVENT = "set-event";
  private static final String SET_NULL_PAYLOAD = "set-null-payload";

  private static ComponentBuildingDefinition.Builder baseDefinition =
      new ComponentBuildingDefinition.Builder().withNamespace(NAME_SPACE);

  @Override
  public void init() {}

  @Override
  public List<ComponentBuildingDefinition> getComponentBuildingDefinitions() {
    LinkedList<ComponentBuildingDefinition> componentBuildingDefinitions = new LinkedList<>();


    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(CONFIG)
                                         .withTypeDefinition(fromType(MunitModule.class))
                                         .withSetterParameterDefinition("name",
                                                                        fromSimpleParameter(MunitModule.MUNIT_MODULE_ID).build())
                                         .withSetterParameterDefinition("disableFlowSources",
                                                                        fromSimpleParameter("disableFlowSources").build())
                                         .withSetterParameterDefinition("excludeFlowSources",
                                                                        fromChildConfiguration(ExcludeFlowSources.class)
                                                                            .withIdentifier("exclude-flow-sources").build())
                                         .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("exclude-flow-sources").withTypeDefinition(fromType(ExcludeFlowSources.class))
            .withSetterParameterDefinition("flows", fromChildConfiguration(List.class).withIdentifier("flows").build())
            .build());
    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("flows").withTypeDefinition(fromType(List.class)).build());
    componentBuildingDefinitions.add(baseDefinition.withIdentifier("flow").withTypeDefinition(fromType(String.class))
        .build());


    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(TEST)
                                         .withTypeDefinition(fromType(TestFlow.class))
                                         .withSetterParameterDefinition("ignore", fromSimpleParameter("ignore").build())
                                         .withSetterParameterDefinition("description", fromSimpleParameter("description").build())
                                         .withSetterParameterDefinition("expectedErrorType",
                                                                        fromSimpleParameter("expectedErrorType").build())
                                         .withSetterParameterDefinition("expectedException",
                                                                        fromSimpleParameter("expectedException").build())
                                         .withSetterParameterDefinition("tags", fromSimpleParameter("tags").build())

                                         .withSetterParameterDefinition("processorChains",
                                                                        fromChildCollectionConfiguration(Object.class).build())
                                         .build());

    componentBuildingDefinitions.add(buildSimpleFlowDefinition(BEFORE_SUITE, BeforeSuite.class));
    componentBuildingDefinitions.add(buildSimpleFlowDefinition(BEFORE_TEST, BeforeTest.class));
    componentBuildingDefinitions.add(buildSimpleFlowDefinition(AFTER_TEST, AfterTest.class));
    componentBuildingDefinitions.add(buildSimpleFlowDefinition(AFTER_SUITE, AfterSuite.class));


    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(SET_NULL_PAYLOAD)
                                         .asPrototype()
                                         .withTypeDefinition(fromType(SetEventProcessor.class))
                                         .withSetterParameterDefinition("payload", fromSimpleParameter("payload").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(SET_EVENT)
                                         .asPrototype()
                                         .withTypeDefinition(fromType(SetEventProcessor.class))
                                         .withSetterParameterDefinition("cloneOriginalEvent",
                                                                        fromSimpleParameter("cloneOriginalEvent").build())
                                         .withSetterParameterDefinition("payload",
                                                                        fromChildConfiguration(Payload.class)
                                                                            .withIdentifier("payload").build())
                                         .withSetterParameterDefinition("attributes",
                                                                        fromChildConfiguration(EventAttributes.class)
                                                                            .withIdentifier("attributes").build())
                                         .withSetterParameterDefinition("error",
                                                                        fromChildConfiguration(UntypedEventError.class)
                                                                            .withIdentifier("error").build())
                                         .withSetterParameterDefinition("variables",
                                                                        fromChildConfiguration(List.class)
                                                                            .withIdentifier("variables").build())
                                         .withSetterParameterDefinition("inboundProperties",
                                                                        fromChildConfiguration(List.class)
                                                                            .withIdentifier("inbound-properties").build())
                                         .withSetterParameterDefinition("sessionProperties",
                                                                        fromChildConfiguration(List.class)
                                                                            .withIdentifier("session-properties").build())
                                         .withSetterParameterDefinition("outboundProperties",
                                                                        fromChildConfiguration(List.class)
                                                                            .withIdentifier("outbound-properties").build())
                                         .withSetterParameterDefinition("inboundAttachments",
                                                                        fromChildConfiguration(List.class)
                                                                            .withIdentifier("inbound-attachments").build())
                                         .withSetterParameterDefinition("outboundAttachments",
                                                                        fromChildConfiguration(List.class)
                                                                            .withIdentifier("outbound-attachments").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier("payload")
                                         .withTypeDefinition(fromType(Payload.class))
                                         .withSetterParameterDefinition("value", fromSimpleParameter("value").build())
                                         .withSetterParameterDefinition("mediaType", fromSimpleParameter("mediaType").build())
                                         .withSetterParameterDefinition("encoding", fromSimpleParameter("encoding").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier("attributes")
                                         .withTypeDefinition(fromType(EventAttributes.class))
                                         .withSetterParameterDefinition("value", fromSimpleParameter("value").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier("error")
                                         .withTypeDefinition(fromType(UntypedEventError.class))
                                         .withSetterParameterDefinition("typeId", fromSimpleParameter("id").build())
                                         .withSetterParameterDefinition("cause", fromSimpleParameter("exception").build())
                                         .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("variables").withTypeDefinition(fromType(List.class)).build());
    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("inbound-properties").withTypeDefinition(fromType(List.class)).build());
    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("session-properties").withTypeDefinition(fromType(List.class)).build());
    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("outbound-properties").withTypeDefinition(fromType(List.class)).build());
    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("inbound-attachments").withTypeDefinition(fromType(List.class)).build());
    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("outbound-attachments").withTypeDefinition(fromType(List.class)).build());

    componentBuildingDefinitions.add(buildGenericPropertyDefinition("variable", Variable.class));
    componentBuildingDefinitions.add(buildGenericPropertyDefinition("inbound-property", Property.class));
    componentBuildingDefinitions.add(buildGenericPropertyDefinition("session-property", Property.class));
    componentBuildingDefinitions.add(buildGenericPropertyDefinition("outbound-property", Property.class));
    componentBuildingDefinitions.add(buildGenericPropertyDefinition("inbound-attachment", Attachment.class));
    componentBuildingDefinitions.add(buildGenericPropertyDefinition("outbound-attachment", Attachment.class));


    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("processorChain"));

    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("behavior"));
    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("execution"));
    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("validation"));

    return componentBuildingDefinitions;
  }

  private ComponentBuildingDefinition buildSimpleFlowDefinition(String identifier, Class typeClass) {
    return baseDefinition.withIdentifier(identifier)
        .withTypeDefinition(fromType(typeClass))
        .withSetterParameterDefinition("description", fromSimpleParameter("description").build())
        .withSetterParameterDefinition("processors", fromChildCollectionConfiguration(Object.class).build())
        .build();
  }

  private ComponentBuildingDefinition buildGenericPropertyDefinition(String identifier, Class clazz) {
    return baseDefinition
        .withIdentifier(identifier)
        .withTypeDefinition(fromType(clazz))
        .withSetterParameterDefinition("key", fromSimpleParameter("key").build())
        .withSetterParameterDefinition("value", fromSimpleParameter("value").build())
        .withSetterParameterDefinition("mediaType", fromSimpleParameter("mediaType").build())
        .withSetterParameterDefinition("encoding", fromSimpleParameter("encoding").build())
        .build();
  }

  private ComponentBuildingDefinition buildDefaultProcessorChainDefinition(String name) {
    return baseDefinition.withIdentifier(name)
        .withTypeDefinition(fromType(AnnotatedObject.class))
        .withObjectFactoryType(MessageProcessorChainObjectFactory.class)
        .withSetterParameterDefinition("messageProcessors", fromChildCollectionConfiguration(Object.class).build())
        .build();
  }

}
