/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.context.plugin;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.common.extension.MunitPlugin;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.context.MuleContextAware;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.*;

/**
 * Loads all the {@link MunitPlugin} that are in the Class Loader, those plugins are set with the {@link MuleContext}
 * (if necessary) but are not initialised
 *
 * @author Mulesoft Inc.
 * @since 3.4
 */
public class MunitPluginFactory {

  private transient Log logger = LogFactory.getLog(this.getClass());

  public static final String META_INF_MUNIT_PLUGIN_PROPERTIES = "META-INF/munit-plugin.properties";

  /**
   * It loads each MUnit Plugin it can find in the current thread classloader
   *
   * @param context the context require to be send to each plugin loaded
   * @return a list containing all the loaded MUnit plugins. Empty if non have been found
   */
  public Collection<MunitPlugin> loadPlugins(MuleContext context) {
    logger.debug("Loading MUnitPlugins...");

    List<MunitPlugin> munitPlugins = new ArrayList<>();
    ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      Enumeration<URL> resources = contextClassLoader.getResources(META_INF_MUNIT_PLUGIN_PROPERTIES);
      while (resources.hasMoreElements()) {
        Properties properties = new Properties();
        Object content = resources.nextElement().getContent();
        properties.load((InputStream) content);
        MunitPlugin plugin = createMunitPlugin(contextClassLoader, properties);
        if (plugin != null) {
          if (plugin instanceof MuleContextAware) {
            ((MuleContextAware) plugin).setMuleContext(context);
          }
          munitPlugins.add(plugin);

          logger.debug("MUnit plugin: " + plugin.getClass().getCanonicalName() + " loaded");
        }
      }
    } catch (IOException e) {
      logger.error("Could not read the Classpath in order to get the plugin configurations");
    }

    return munitPlugins;
  }

  private MunitPlugin createMunitPlugin(ClassLoader contextClassLoader, Properties properties) {
    String property = properties.getProperty("plugin.className");
    try {
      if (property != null && !property.isEmpty()) {
        logger.debug("Attempting to load MUnit plugin: " + property);
        return (MunitPlugin) contextClassLoader.loadClass(property).newInstance();
      }
    } catch (Throwable e) {
      logger.error("The class " + property + " could not be load");
    }
    return null;
  }
}
