/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.context.plugin;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.common.extension.MunitPlugin;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.core.api.MuleContext;

import java.util.Collection;
import java.util.Collections;

import static org.mule.runtime.api.util.Preconditions.checkNotNull;

/**
 * The class is a container of MUnit plugins and through it you should handle the plugins lifecycle
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitPluginManager {

  private transient Log logger = LogFactory.getLog(this.getClass());

  private MuleContext context;
  private MunitPluginFactory pluginFactory;

  private Collection<MunitPlugin> plugins = Collections.emptyList();

  public MunitPluginManager() {
    this.pluginFactory = new MunitPluginFactory();
  }

  public void setMuleContext(MuleContext context) {
    checkNotNull(context, "The context must not be null");
    this.context = context;
  }

  public void setPluginFactory(MunitPluginFactory pluginFactory) {
    checkNotNull(pluginFactory, "The plugin factory must not be null");
    this.pluginFactory = pluginFactory;
  }

  public void loadAndInitialisePlugins() throws InitialisationException {
    loadPlugins();
    initialisePlugins();
  }

  public void loadPlugins() {
    logger.debug("Loading MUnit plugins...");

    if (null == context) {
      throw new IllegalStateException("The plugin manager requires a mule context and it's currently null");
    }

    plugins = pluginFactory.loadPlugins(context);
  }

  public void initialisePlugins() throws InitialisationException {
    logger.debug("Initializing MUnit plugins...");
    for (MunitPlugin plugin : plugins) {
      plugin.initialise();
      logger.debug(plugin.getClass().getName() + " MUnit plugin initialised");
    }
  }

  public void startPlugins() throws MuleException {
    logger.debug("Starting MUnit plugins...");
    for (MunitPlugin plugin : plugins) {
      plugin.start();
      logger.debug(plugin.getClass().getName() + " MUnit plugin started");
    }
  }

  public void stopPlugins() throws MuleException {
    logger.debug("Stopping MUnit plugins...");
    for (MunitPlugin plugin : plugins) {
      plugin.stop();
      logger.debug(plugin.getClass().getName() + " MUnit plugin stop");
    }
  }

  public void disposePlugins() {
    logger.debug("Disposing MUnit plugins...");
    for (MunitPlugin plugin : plugins) {
      plugin.dispose();
      logger.debug(plugin.getClass().getName() + " MUnit plugin disposed");
    }
  }



}
