/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model.builders;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.AFTER_SUITE;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.AFTER_TEST;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.BEFORE_SUITE;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.BEFORE_TEST;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.CONFIG;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.TEST;
import static org.mule.munit.runner.config.MunitXmlNamespaceInfoProvider.NAME_SPACE;
import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.runner.model.Suite;
import org.mule.munit.runner.model.Test;
import org.mule.munit.runner.output.DefaultOutputHandler;
import org.mule.munit.runner.output.TestOutputHandler;
import org.mule.munit.runner.flow.AfterSuite;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeSuite;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.munit.runner.flow.TestFlow;
import org.mule.munit.runner.remote.api.notifiers.DummySuiteRunEventListener;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.meta.AnnotatedObject;

/**
 * Creates a MUnit Suite and its Tests
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SuiteBuilder {

  protected String suitePath;
  private ConfigurationComponentLocator configurationComponentLocator;

  protected Set<String> tags;
  protected List<String> testNames;
  protected TestOutputHandler outputHandler;
  protected SuiteRunEventListener suiteRunEventListener;

  public SuiteBuilder(String suitePath, ConfigurationComponentLocator configurationComponentLocator) {
    checkArgument(StringUtils.isNotBlank(suitePath), "The suitePath must not be null nor empty");
    checkNotNull(configurationComponentLocator, "The configuration component locator must not be null");

    this.suitePath = suitePath;
    this.outputHandler = new DefaultOutputHandler();
    this.suiteRunEventListener = new DummySuiteRunEventListener();
    this.configurationComponentLocator = configurationComponentLocator;
  }

  public SuiteBuilder withTestNames(List<String> testNames) {
    this.testNames = testNames;
    return this;
  }

  public SuiteBuilder withTags(Set<String> tags) {
    this.tags = tags;
    return this;
  }

  public SuiteBuilder withOutputHandler(TestOutputHandler outputHandler) {
    checkNotNull(outputHandler, "The outputHandler must not be null");
    this.outputHandler = outputHandler;
    return this;
  }

  public SuiteBuilder withSuiteRunnerEventListener(SuiteRunEventListener suiteRunEventListener) {
    checkNotNull(suiteRunEventListener, "The suiteRunEventListener must not be null");
    this.suiteRunEventListener = suiteRunEventListener;
    return this;
  }

  /**
   * Builds the Suite with a particular suite name, based on the mule context
   * 
   * @return The Suite Object
   */
  public Suite build() {
    MunitModule munitModule = lookupMunitModule();

    Suite suite = new Suite(suitePath);
    suite.setOutputHandler(outputHandler);
    suite.setSuiteRunEventListener(suiteRunEventListener);

    suite.setBeforeSuite(lookupBeforeSuite());
    createTests(munitModule).forEach(suite::addTest);
    suite.setAfterSuite(lookupAfterSuite());

    return suite;
  }

  private List<Test> createTests(MunitModule munitModule) {
    BeforeTest before = lookupBeforeTest();
    Collection<TestFlow> testFlows = lookupTests();
    AfterTest after = lookupAfterTest();

    TestRunFilter filter = new TestRunFilter();

    List<Test> tests = new ArrayList<>();
    tests.addAll(testFlows.stream()
        .filter(testFlow -> filter.shouldRunTest(testFlow, testNames, tags))
        .map(testFlow -> test(before, testFlow, after, munitModule))
        .collect(Collectors.toList()));

    return tests;
  }

  /**
   * Create a test
   *
   * @param beforeTest MUnit flow to be run before the test
   * @param test MUnit Flow that represents the test
   * @param afterTest MUnit flow to be run after the test
   * @return The Test Object
   */
  protected Test test(BeforeTest beforeTest, TestFlow test, AfterTest afterTest, MunitModule munitModule) {
    return new Test(beforeTest, test, afterTest, outputHandler, munitModule);
  }

  private MunitModule lookupMunitModule() {
    String componentIdentifier = NAME_SPACE + ":" + CONFIG;
    List<AnnotatedObject> annotatedObjects =
        configurationComponentLocator.find(buildFromStringRepresentation(componentIdentifier));

    if (!annotatedObjects.isEmpty()) {
      return (MunitModule) annotatedObjects.get(0);
    }
    throw new IllegalStateException("Missing element [" + componentIdentifier + "]");
  }

  private BeforeSuite lookupBeforeSuite() {
    String componentIdentifier = NAME_SPACE + ":" + BEFORE_SUITE;
    List<AnnotatedObject> annotatedObjects =
        configurationComponentLocator.find(buildFromStringRepresentation(componentIdentifier));

    if (!annotatedObjects.isEmpty()) {
      return (BeforeSuite) annotatedObjects.get(0);
    }
    return null;
  }

  private AfterSuite lookupAfterSuite() {
    String componentIdentifier = NAME_SPACE + ":" + AFTER_SUITE;
    List<AnnotatedObject> annotatedObjects =
        configurationComponentLocator.find(buildFromStringRepresentation(componentIdentifier));

    if (!annotatedObjects.isEmpty()) {
      return (AfterSuite) annotatedObjects.get(0);
    }
    return null;
  }

  private BeforeTest lookupBeforeTest() {
    String componentIdentifier = NAME_SPACE + ":" + BEFORE_TEST;
    List<AnnotatedObject> annotatedObjects =
        configurationComponentLocator.find(buildFromStringRepresentation(componentIdentifier));

    if (!annotatedObjects.isEmpty()) {
      return (BeforeTest) annotatedObjects.get(0);
    }
    return null;
  }

  private AfterTest lookupAfterTest() {
    String componentIdentifier = NAME_SPACE + ":" + AFTER_TEST;
    List<AnnotatedObject> annotatedObjects =
        configurationComponentLocator.find(buildFromStringRepresentation(componentIdentifier));

    if (!annotatedObjects.isEmpty()) {
      return (AfterTest) annotatedObjects.get(0);
    }
    return null;
  }

  private Collection<TestFlow> lookupTests() {
    String componentIdentifier = NAME_SPACE + ":" + TEST;
    List<AnnotatedObject> annotatedObjects =
        configurationComponentLocator.find(buildFromStringRepresentation(componentIdentifier));

    return annotatedObjects.stream().map(annotatedObject -> (TestFlow) annotatedObject).collect(Collectors.toList());
  }


}
