/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.processors;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.util.MunitExpressionWrapper;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.core.api.InternalEvent;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.i18n.CoreMessages;
import org.mule.runtime.core.api.exception.MessagingException;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.api.meta.AbstractAnnotatedObject;

/**
 * Generic Munit Processor.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class MunitProcessor extends AbstractAnnotatedObject implements Initialisable, Processor {

  @Inject
  protected MuleContext muleContext;

  protected MunitExpressionWrapper expressionWrapper;

  public void initialise() throws InitialisationException {
    expressionWrapper = new MunitExpressionWrapper(muleContext.getExpressionManager());
  }

  /**
   * Executes the message processor code. In case of an assertion error it throws a new exception with the mule stack trace
   * 
   * @param event The mule event to be processed.
   * @return The result mule event
   * @throws MuleException In case of error. If the assertion fails, it throws an {@link AssertionError}
   * 
   */
  public InternalEvent process(InternalEvent event) throws MuleException {
    try {
      return doProcess(event);
    } catch (AssertionError error) {
      AssertionError exception = new AssertionError(getMessage(error));
      throw exception;
    } catch (Exception e) {
      throw new MessagingException(CoreMessages.failedToInvoke(getProcessor()), event, e);
    }
  }

  /**
   * The method that do the actual process
   *
   * @param event * The mule Event
   * @return the processed event
   */
  protected abstract InternalEvent doProcess(InternalEvent event);

  /**
   * @return The name of the processor
   */
  protected abstract String getProcessor();

  public void setMuleContext(MuleContext context) {
    this.muleContext = context;
  }

  private String getMessage(AssertionError error) {
    String message = error.getMessage();
    if (StringUtils.isEmpty(message)) {
      return this.getProcessor();
    }
    return message;
  }
}
