/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote.api.server;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.core.api.MuleContext;

/**
 * The goal of this class is to offer an entry point to the container through the munit-runner plugin with the objective to run
 * MUnit Test Suites
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerServer implements Runnable, Lifecycle {

  private static final int SOCKET_TIMEOUT_MILLIS = 10000;
  protected static final String MUNIT_SERVER_PORT = "munit.server.port";

  private static transient Log log = LogFactory.getLog(RunnerServer.class);

  @Inject
  private MuleContext muleContext;

  @Inject
  private ConfigurationComponentLocator componentLocator;

  private int port;
  private boolean keepRunning = true;
  private ServerSocket providerSocket = null;

  public RunnerServer() {}

  public void run() {
    try {
      providerSocket = createServerSocket();

      do {
        log.info("Waiting for client connection ");
        Socket connection = providerSocket.accept();
        log.info("Client connection received from " + connection.getInetAddress().getHostName() + " - " + keepRunning);

        ObjectOutput out = new ObjectOutputStream(connection.getOutputStream());
        ObjectInput in = new ObjectInputStream(connection.getInputStream());

        handleClientMessage(in, out);

      } while (keepRunning);

    } catch (SocketTimeoutException timeoutException) {
      log.debug("MUnit server time out");
      if (keepRunning) {
        log.error("Client connection timeout after " + String.valueOf(SOCKET_TIMEOUT_MILLIS) + " milliseconds");
      }
    } catch (IOException ioException) {
      if (providerSocket != null && providerSocket.isClosed()) {
        if (keepRunning) {
          log.warn("Kill signal received before accept timeout in port [" + port + "]", ioException);
        } else {
          log.debug("Shut down signal received MUnit server running in port [" + port + "]...");
        }
      } else {
        log.error("Failed to start MUnit server in port [" + port + "]", ioException);
      }
    } catch (ClassNotFoundException e) {
      log.error("Fail to deserialize message.", e);
    } finally {
      try {
        log.debug("Shutting down MUnit server running in port [" + port + "]...");

        if (null != providerSocket) {
          providerSocket.close();
        }
        keepRunning = false;
        log.debug("MUnit server shutdown");
      } catch (IOException ioException) {
        log.debug("MUnit server error during shutdown.");
      }
    }
  }

  protected ServerSocket createServerSocket() throws IOException {
    port = Integer.parseInt(System.getProperty(MUNIT_SERVER_PORT));
    ServerSocket providerSocket = new ServerSocket(port, 10);
    providerSocket.setSoTimeout(SOCKET_TIMEOUT_MILLIS);
    log.debug("MUnit server started listening in port [" + port + "]...");
    return providerSocket;
  }

  protected void handleClientMessage(ObjectInput in, ObjectOutput out) throws IOException, ClassNotFoundException {
    RunMessageHandler commander = new RunMessageHandler(in, out, componentLocator);
    commander.handle();
  }

  protected void setMuleContext(MuleContext muleContext) {
    this.muleContext = muleContext;
  }

  protected boolean isKeepRunning() {
    return keepRunning;
  }

  @Override
  public void initialise() throws InitialisationException {
    log.debug("Initializing MUnit server...");
    muleContext.getSchedulerService().ioScheduler().schedule(this, 0, TimeUnit.SECONDS);
  }

  @Override
  public void start() throws MuleException {

  }

  @Override
  public void stop() throws MuleException {
    log.debug("Stop signal received, shooting down MUnit server...");
    keepRunning = false;
  }

  @Override
  public void dispose() {
    log.debug("Dispose signal received, shooting down MUnit server...");
    try {
      keepRunning = false;
      if (providerSocket != null) {
        providerSocket.close();
      }
    } catch (IOException e) {
      log.warn("Error when sending kill signal to MUnit server", e);
    }
  }

}
