/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.processors;

import static org.apache.commons.lang3.Validate.notNull;

import java.io.IOException;
import java.io.Serializable;
import java.util.Collections;
import java.util.List;

import javax.activation.DataHandler;
import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.event.EventBuilder;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.model.Attachment;
import org.mule.munit.common.model.EventAttributes;
import org.mule.munit.common.model.EventError;
import org.mule.munit.common.model.NullObject;
import org.mule.munit.common.model.Payload;
import org.mule.munit.common.model.Property;
import org.mule.munit.common.model.UntypedEventError;
import org.mule.munit.common.model.Variable;
import org.mule.munit.common.util.IOUtils;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.DataTypeBuilder;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.privileged.PrivilegedMuleContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Sets the payload
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SetEventProcessor extends MunitProcessor {

  private static final String MEDIA_TYPE_FIELD = "Media Type";
  private static final String ENCODING_FIELD = "Encoding";
  private static final String KEY_FIELD = "Key";

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private Boolean cloneOriginalEvent = false;
  private org.mule.munit.common.model.Event event = new org.mule.munit.common.model.Event();


  @Inject
  PrivilegedMuleContext muleContext;

  @Override
  protected String getProcessor() {
    return "set";
  }

  @Override
  protected CoreEvent doProcess(CoreEvent incomingEvent) {

    EventBuilder builder = cloneOriginalEvent ? new EventBuilder(incomingEvent) : new EventBuilder(incomingEvent.getContext());

    evaluatePayload(event.getPayload(), incomingEvent, builder);

    evaluateMediaType(event.getPayload(), incomingEvent, builder);

    evaluateAttributes(event.getAttributes(), incomingEvent, builder);

    evaluateErrorType(event.getError(), incomingEvent, builder);

    evaluateVariables(event.getVariables(), incomingEvent, builder);

    evaluatePropertiesAndAttachments(event, incomingEvent, builder);

    return (CoreEvent) builder.build();
  }

  private void evaluatePropertiesAndAttachments(org.mule.munit.common.model.Event returnEvent, CoreEvent incomingEvent,
                                                EventBuilder builder) {
    evaluateProperties(returnEvent.getSessionProperties(), returnEvent.getInboundProperties(),
                       returnEvent.getOutboundProperties(), incomingEvent, builder);
    evaluateAttachments(returnEvent.getInboundAttachments(), returnEvent.getOutboundAttachments(), incomingEvent, builder);
    logWarningIfPropertiesOrAttachmentsPresent(returnEvent.getInboundAttachments(), returnEvent.getOutboundAttachments(),
                                               returnEvent.getSessionProperties(), returnEvent.getInboundProperties(),
                                               returnEvent.getOutboundProperties());
  }

  private void evaluateAttachments(List<Attachment> inboundAttachments, List<Attachment> outboundAttachments,
                                   CoreEvent incomingEvent,
                                   EventBuilder builder) {
    if (null != inboundAttachments) {
      if (inboundAttachments.isEmpty()) {
        builder.withInboundAttachments(Collections.emptyMap());
      } else {
        inboundAttachments.forEach(a -> builder.addInboundAttachment(evaluateAsString(incomingEvent, a.getKey(), KEY_FIELD),
                                                                     getDataHandler(incomingEvent, a)));
      }
    }

    if (null != outboundAttachments) {
      if (outboundAttachments.isEmpty()) {
        builder.withOutboundAttachments(Collections.emptyMap());
      } else {
        outboundAttachments.forEach(a -> builder.addOutboundAttachment(evaluateAsString(incomingEvent, a.getKey(), KEY_FIELD),
                                                                       getDataHandler(incomingEvent, a)));
      }
    }
  }

  private void evaluateProperties(List<Property> sessionProperties, List<Property> inboundProperties,
                                  List<Property> outboundProperties, CoreEvent incomingEvent, EventBuilder builder) {
    if (null != sessionProperties) {
      if (sessionProperties.isEmpty()) {
        builder.clearSessionProperties();
      } else {
        sessionProperties.forEach(p -> builder.addSessionProperty(evaluateAsString(incomingEvent, p.getKey(), KEY_FIELD),
                                                                  evaluate(incomingEvent, p.getValue()),
                                                                  evaluateMediaTypeExpression(incomingEvent, p.getMediaType()),
                                                                  evaluateAsString(incomingEvent, p.getEncoding(),
                                                                                   ENCODING_FIELD)));
      }
    }

    if (null != inboundProperties) {
      if (inboundProperties.isEmpty()) {
        builder.withInboundProperties(Collections.emptyMap());
      } else {
        inboundProperties.forEach(p -> builder.addInboundProperty(evaluateAsString(incomingEvent, p.getKey(), KEY_FIELD),
                                                                  (Serializable) evaluate(incomingEvent, p.getValue()),
                                                                  evaluateMediaTypeExpression(incomingEvent, p.getMediaType()),
                                                                  evaluateAsString(incomingEvent, p.getEncoding(),
                                                                                   ENCODING_FIELD)));
      }
    }

    if (null != outboundProperties) {
      if (outboundProperties.isEmpty()) {
        builder.withOutboundProperties(Collections.emptyMap());
      } else {
        outboundProperties.forEach(p -> builder.addOutboundProperty(evaluateAsString(incomingEvent, p.getKey(), KEY_FIELD),
                                                                    (Serializable) evaluate(incomingEvent, p.getValue()),
                                                                    evaluateMediaTypeExpression(incomingEvent, p.getMediaType()),
                                                                    evaluateAsString(incomingEvent, p.getEncoding(),
                                                                                     ENCODING_FIELD)));
      }
    }
  }

  private void evaluateVariables(List<Variable> variables, CoreEvent incomingEvent, EventBuilder builder) {
    if (null != variables) {
      if (variables.isEmpty()) {
        builder.withVariables(Collections.emptyMap());
      } else {
        variables.forEach(v -> builder.addVariable(evaluateAsString(incomingEvent, v.getKey(), KEY_FIELD),
                                                   evaluate(incomingEvent, v.getValue()),
                                                   evaluateMediaTypeExpression(incomingEvent, v.getMediaType()),
                                                   evaluateAsString(incomingEvent, v.getEncoding(), ENCODING_FIELD)));
      }
    }
  }

  private void evaluatePayload(Payload payload, CoreEvent incomingEvent, EventBuilder builder) {
    if (!(payload.getValue() instanceof NullObject)) {
      builder.withPayload(evaluate(incomingEvent, payload.getValue()));
    }
  }

  private void evaluateMediaType(Payload payload, CoreEvent incomingEvent, EventBuilder builder) {
    DataTypeBuilder dataTypeBuilder = DataType.builder();
    MediaType mediaType = null;
    if (cloneOriginalEvent && StringUtils.isNotBlank(payload.getMediaType())) {
      dataTypeBuilder.mediaType(evaluateMediaTypeExpression(incomingEvent, payload.getMediaType()));
    }

    if (!cloneOriginalEvent) {
      dataTypeBuilder.mediaType(evaluateMediaTypeExpression(incomingEvent, payload.getMediaType()));
    }

    if (StringUtils.isNotBlank(payload.getEncoding())) {
      dataTypeBuilder.charset(evaluateAsString(incomingEvent, payload.getEncoding(), ENCODING_FIELD));
    }

    if ((cloneOriginalEvent && StringUtils.isNotBlank(payload.getMediaType())) || !cloneOriginalEvent ||
        StringUtils.isNotBlank(payload.getEncoding())) {
      mediaType = dataTypeBuilder.build().getMediaType();
    }
    if (mediaType != null) {
      builder.withMediaType(mediaType);
    }
  }

  private void evaluateAttributes(EventAttributes attributes, CoreEvent incomingEvent, EventBuilder builder) {
    try {
      if (attributes.getValue() != null && !(attributes.getValue() instanceof NullObject)) {
        builder.withAttributes(evaluate(incomingEvent, attributes.getValue()));
      }

    } catch (ClassCastException e) {
      throw new MunitError("Attributes evaluation failed", e);
    }
  }

  private void evaluateErrorType(EventError error, CoreEvent incomingEvent, EventBuilder builder) {
    ErrorType errorType = null;
    String errorTypeId = evaluateAsString(incomingEvent, error.getTypeId(), "Error Type Id");
    if (StringUtils.isNotBlank(errorTypeId) && null != evaluate(incomingEvent, error.getCause())) {
      ComponentIdentifier componentIdentifier = ComponentIdentifier.buildFromStringRepresentation(errorTypeId);
      errorType = muleContext.getErrorTypeLocator().lookupComponentErrorType(componentIdentifier,
                                                                             evaluateErrorCause(error, incomingEvent));
    }
    if (null != errorType) {
      builder.withError(errorType, evaluateErrorCause(error, incomingEvent));
    }
  }

  private Throwable evaluateErrorCause(EventError error, CoreEvent incomingEvent) {
    try {
      return (Throwable) evaluate(incomingEvent, error.getCause());
    } catch (ClassCastException e) {
      throw new MunitError(String.format("Error cause '%s' should be Throwable", error.getCause()), e);
    }
  }

  private String evaluateAsString(CoreEvent event, Object expression, String name) {
    return expressionWrapper.evaluateAsStringIfExpression(event, expression, name);
  }

  protected Object evaluate(CoreEvent event, Object possibleExpression) {
    return expressionWrapper.evaluateIfExpression(event, possibleExpression).getValue();
  }

  private String evaluateMediaTypeExpression(CoreEvent event, Object mediaTypeExpression) {
    if (mediaTypeExpression == null) {
      return MediaType.ANY.toString();
    }
    return expressionWrapper.evaluateNotNullString(event, mediaTypeExpression, MEDIA_TYPE_FIELD);
  }

  private DataHandler getDataHandler(CoreEvent event, Attachment attachment) {
    MediaType contentType = DataType.builder()
        .mediaType(evaluateMediaTypeExpression(event, attachment.getMediaType()))
        .charset(expressionWrapper.evaluateAsStringIfExpression(event, attachment.getEncoding(), "Encoding"))
        .build().getMediaType();

    try {
      return IOUtils.toDataHandler(attachment.getKey(),
                                   expressionWrapper.evaluateIfExpression(event, attachment.getValue()).getValue(),
                                   contentType);
    } catch (IOException e) {
      // TODO handle this properly
      throw new RuntimeException(e);
    }
  }

  public void setCloneOriginalEvent(Boolean cloneOriginalEvent) {
    this.cloneOriginalEvent = cloneOriginalEvent;
  }

  public void setPayload(Payload payload) {
    event.setPayload(payload);
  }

  public void setAttributes(EventAttributes attributes) {
    event.setAttributes(attributes);
  }

  public void setVariables(List<Variable> variables) {
    notNull(variables, "Variables can not be null");
    event.setVariables(variables);
  }

  public void setSessionProperties(List<Property> sessionProperties) {
    notNull(sessionProperties, "Session properties can not be null");
    event.setSessionProperties(sessionProperties);
  }

  public void setInboundProperties(List<Property> inboundProperties) {
    notNull(inboundProperties, "Inbound properties can not be null");
    event.setInboundProperties(inboundProperties);
  }

  public void setOutboundProperties(List<Property> outboundProperties) {
    notNull(outboundProperties, "Outbound properties can not be null");
    event.setOutboundProperties(outboundProperties);
  }

  public void setInboundAttachments(List<Attachment> inboundAttachments) {
    notNull(inboundAttachments, "Inbound attachments can not be null");
    event.setInboundAttachments(inboundAttachments);
  }

  public void setOutboundAttachments(List<Attachment> outboundAttachments) {
    notNull(outboundAttachments, "outbound attachments can not be null");
    event.setOutboundAttachments(outboundAttachments);
  }

  public void setError(UntypedEventError error) {
    event.setError(error);
  }

  protected void setMuleContext(PrivilegedMuleContext muleContext) {
    this.muleContext = muleContext;
  }

  private void logWarningIfPropertiesOrAttachmentsPresent(List<Attachment> inboundAttachments,
                                                          List<Attachment> outboundAttachments,
                                                          List<Property> sessionProperties,
                                                          List<Property> inboundProperties, List<Property> outboundProperties) {
    boolean hasAttachments = inboundAttachments != null || outboundAttachments != null;
    boolean hasProperties =
        inboundProperties != null || outboundProperties != null || sessionProperties != null;

    if (hasAttachments || hasProperties) {
      if (logger.isDebugEnabled()) {
        logger.warn("Setting properties/attachments will only work in Mule 4 if the compatibility module is present." +
            " If not, they will have no effect since they are meant for Mule versions 3.x");
      } else {
        logger.warn("Setting elements that only work for Mule versions 3.x ");
      }
    }
  }

}
