/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.munit.common.protocol.message.TestStatus.ERROR;
import static org.mule.munit.common.protocol.message.TestStatus.FAILURE;
import static org.mule.munit.common.protocol.message.TestStatus.SUCCESS;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.event.EventBuilder;
import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.common.protocol.message.TestStatus;
import org.mule.munit.common.util.StackTraceUtil;
import org.mule.munit.runner.flow.AfterSuite;
import org.mule.munit.runner.flow.BeforeSuite;
import org.mule.munit.runner.flow.SimpleFlow;
import org.mule.munit.runner.remote.api.notifiers.DummySuiteRunEventListener;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * MUnit Suite. This class is part of the execution model of MUnit.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class Suite {

  protected transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private String path;
  private BeforeSuite beforeSuiteFlow;
  private List<Test> tests;
  private AfterSuite afterSuiteFlow;

  private SuiteRunEventListener suiteRunEventListener;

  public Suite(String path) {
    this.path = path;
    this.tests = new ArrayList<>();

    this.suiteRunEventListener = new DummySuiteRunEventListener();
  }

  public void setSuiteRunEventListener(SuiteRunEventListener suiteRunEventListener) {
    checkNotNull(suiteRunEventListener, "The suiteRunEventListener must not be null");
    this.suiteRunEventListener = suiteRunEventListener;
  }

  public String getPath() {
    return path;
  }

  public void setBeforeSuite(BeforeSuite beforeSuite) {
    beforeSuiteFlow = beforeSuite;
  }

  public void addTest(Test test) {
    tests.add(test);
  }

  public void setAfterSuite(AfterSuite afterSuite) {
    afterSuiteFlow = afterSuite;
  }

  public int getNumberOfTests() {
    return tests.size();
  }

  public boolean allTestsIgnored() {
    return tests.stream().allMatch(Test::isIgnore);
  }

  /**
   * Runs all the tests of the suite
   *
   * @return The Result of the suite execution
   * @throws Exception If the suite failed for one reason.
   */
  public SuiteResult run() throws Exception {
    SuiteResult result = new SuiteResult(path);
    try {
      runBeforeSuite();

      for (Test test : tests) {
        suiteRunEventListener.notifyTestStart(test.getName(), test.getDescription());

        TestResult testResult = test.run();
        result.add(testResult);

        suiteRunEventListener.notifyTestEnd(testResult.getName(), testResult.getStackTrace(), testResult.getStatus(),
                                            testResult.getElapsedTime());
      }
    } catch (Throwable e) {
      logger.error("Could not Run the suite: " + path, e);
    } finally {
      runAfterSuite();
    }
    return result;
  }

  private void runBeforeSuite() throws Throwable {
    String flowName = StringUtils.EMPTY;
    try {
      logger.debug("Executing Before Suite scope...");
      if (null != beforeSuiteFlow) {
        flowName = beforeSuiteFlow.getName();
        Event event = new EventBuilder(beforeSuiteFlow.getLocation()).build();
        suiteRunEventListener.notifyBeforeSuiteStart(flowName);
        process(beforeSuiteFlow, event);
        suiteRunEventListener.notifyBeforeSuiteEnd(flowName, StringUtils.EMPTY, SUCCESS);
      }
    } catch (MuleRuntimeException | MuleException e) {
      Throwable cause = e.getCause();
      handleBeforeSuiteEndByException(flowName, cause);
      throw cause;
    } catch (Throwable e) {
      handleBeforeSuiteEndByException(flowName, e);
      throw e;
    }
  }

  private void runAfterSuite() {
    String flowName = StringUtils.EMPTY;
    try {
      logger.debug("Executing After Suite scope...");
      if (null != afterSuiteFlow) {
        flowName = afterSuiteFlow.getName();
        Event event = new EventBuilder(afterSuiteFlow.getLocation()).build();
        suiteRunEventListener.notifyAfterSuiteStart(flowName);
        process(afterSuiteFlow, event);
        suiteRunEventListener.notifyAfterSuiteEnd(flowName, StringUtils.EMPTY, SUCCESS);
      }
    } catch (MuleRuntimeException | MuleException e) {
      Throwable cause = e.getCause();
      handleAfterSuiteEndByException(flowName, cause);
    } catch (Throwable e) {
      handleAfterSuiteEndByException(flowName, e);
    }
  }

  protected void handleBeforeSuiteEndByException(String flowName, Throwable cause) {
    String message = String.format("Before suite %s execution failed", flowName);
    logger.error(message, cause);
    String stackTrace = getStackTrace(cause, message);
    suiteRunEventListener.notifyBeforeSuiteEnd(flowName, stackTrace, getStatus(cause));
  }

  private void handleAfterSuiteEndByException(String flowName, Throwable cause) {
    String message = String.format("After suite %s execution failed", flowName);
    logger.error(message, cause);
    String stackTrace = getStackTrace(cause, message);
    suiteRunEventListener.notifyAfterSuiteEnd(flowName, stackTrace, getStatus(cause));
  }

  protected void process(SimpleFlow simpleFlow, Event event) throws Throwable {
    try {
      simpleFlow.execute(event).get();
    } catch (ExecutionException e) {
      throw e.getCause();
    }
  }

  private String getStackTrace(Throwable cause, String message) {
    RuntimeException e = new RuntimeException(message, cause);
    return StackTraceUtil.getStackTrace(e);
  }

  private TestStatus getStatus(Throwable e) {
    return e instanceof AssertionError ? FAILURE : ERROR;
  }

}
