/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote.api.notifiers;


import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.repeat;

import java.io.PrintStream;

import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.message.TestStatus;

/**
 * <p>
 * Prints friendly messages of the test results through a given {@link PrintStream}
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class StreamNotifier implements RunEventListener {

  private static final String SUITE_TOKEN = "=";
  private static final String TEST_TOKEN = "+";

  private PrintStream out;
  private int tests = 0;
  private int failures = 0;
  private int errors = 0;
  private int skipped = 0;

  public StreamNotifier(PrintStream out) {
    this.out = out;
  }

  @Override
  public void notifyTestStart(String name, String description) {
    String descriptionMessage = isBlank(description) ? EMPTY : " - " + description;
    String testMessage = TEST_TOKEN + " Running test: " + name + descriptionMessage + repeat(" ", 10) + TEST_TOKEN;
    String title = repeat(TEST_TOKEN, testMessage.length());
    out.println(title);
    out.println(testMessage);
    out.println(title);
    out.flush();
  }

  @Override
  public void notifySuiteStart(String suite, int numberOfTests) {
    String suiteMessage = repeat(SUITE_TOKEN, 12) + "  Running  " + suite + " suite " + repeat(SUITE_TOKEN, 12);
    String title = repeat(SUITE_TOKEN, suiteMessage.length());
    out.println(title);
    out.println(suiteMessage);
    out.println(title);
    out.flush();
  }

  @Override
  public void notifyTestEnd(String name, String stackTrace, TestStatus status, long elapsedTime) {
    tests++;
    String message = EMPTY;
    switch (status) {
      case SUCCESS:
        message = "SUCCESS - Test " + name + " finished successfully";
        break;
      case ERROR:
        message = "ERROR - The test " + name + " finished with an error";
        errors++;
        break;
      case FAILURE:
        message = "FAILURE - The test " + name + " finished with a failure";
        failures++;
        break;
      case IGNORED:
        message = "SKIPPED - Test " + name + " was Skipped";
        skipped++;
        break;
    }
    message = TEST_TOKEN + " " + message + " - Time elapsed: " + elapsedTime + "ms " + TEST_TOKEN;
    out.println(repeat(TEST_TOKEN, message.length()));
    out.println(message);
    if (StringUtils.isNotBlank(stackTrace)) {
      out.println(stackTrace);
    }
    out.println(repeat(TEST_TOKEN, message.length()));
    out.flush();
  }

  @Override
  public void notifySuiteEnd(String suite, long elapsedTime) {
    out.println();
    String title = "Number of tests run: " + tests + " - Failed: " + failures + " - Errors: " + errors + " - Skipped: " + skipped
        + " - Time elapsed: " + elapsedTime + "ms";
    String titleFrame = StringUtils.repeat(SUITE_TOKEN, title.length());
    out.println(titleFrame);
    out.println(title);
    out.println(titleFrame);
    out.flush();
  }

}
