/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.flow;

import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;
import static junit.framework.Assert.fail;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.junit.ComparisonFailure;

import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.util.MunitExpressionWrapper;
import org.mule.munit.runner.processors.EnableFlowSources;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.component.execution.ExecutionResult;
import org.mule.runtime.api.component.execution.InputEvent;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerConfig;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.api.construct.Pipeline;
import org.mule.runtime.core.api.el.ExtendedExpressionManager;
import org.mule.runtime.core.api.exception.SingleErrorTypeMatcher;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.core.api.functional.Either;
import org.mule.runtime.core.api.lifecycle.LifecycleUtils;
import org.mule.runtime.core.api.source.MessageSource;
import org.mule.runtime.core.privileged.processor.CompositeProcessorChainRouter;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * The Test Flow
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class TestFlow extends CompositeProcessorChainRouter {

  public static final String MUNIT_TEST_TIMEOUT_PROPERTY = "munit.test.timeout";

  protected static final transient Logger logger = LoggerFactory.getLogger(TestFlow.class);

  private static final int DEFAULT_TIMEOUT = 60000;

  @Inject
  private ErrorTypeRepository errorTypeRepository;

  @Inject
  private ExtendedExpressionManager extendedExpressionManager;

  @Inject
  protected Optional<MunitModule> munitModule;

  @Inject
  private SchedulerService schedulerService;

  private static final String TAG_SEPARATOR = ",";
  public static final String NO_TAG_TOKEN = "NO_TAG";

  /**
   * <p>
   * The description of the test
   * </p>
   */
  private String description;

  /**
   * <p>
   * Determines if the test has to be ignored
   * </p>
   */
  private boolean ignore;

  /**
   * <p>
   * The error type id that is expected
   * </p>
   */
  private String expectedErrorType;

  /**
   * <p>
   * The name of the exception that is expected
   * </p>
   */
  private String expectedException;

  /**
   * <p>
   * The tags that the test has
   * </p>
   */
  private String tags;

  /**
   * <p>
   * List the flow whose sources should be enabled
   * </p>
   */
  private EnableFlowSources enableFlowSources;

  /**
   * <p>
   * List the sources which should be enabled
   * </p>
   */
  private List<MessageSource> flowSources = emptyList();
  private Scheduler scheduler;

  public boolean isIgnore() {
    return ignore;
  }

  public void setIgnore(boolean ignore) {
    this.ignore = ignore;
  }

  public String getExpectedErrorType() {
    return expectedErrorType;
  }

  public void setExpectedErrorType(String expectedErrorType) {
    this.expectedErrorType = expectedErrorType;
  }

  public String getExpectedException() {
    return expectedException;
  }

  public void setExpectedException(String expectedException) {
    this.expectedException = expectedException;
  }

  public Set<String> getTags() {
    if (StringUtils.isBlank(tags)) {
      return Collections.emptySet();
    }
    Set<String> tagSet = Stream.of(tags.split(TAG_SEPARATOR)).collect(Collectors.toSet());
    if (tagSet.stream().anyMatch(tag -> tag.trim().equalsIgnoreCase(NO_TAG_TOKEN))) {
      throw new IllegalArgumentException("The tag '" + NO_TAG_TOKEN + "' is invalid since it's a keyword.");
    }
    return tagSet;
  }

  public void setTags(String tags) {
    this.tags = tags;
  }

  public String getDescription() {
    return description;
  }

  public void setDescription(String description) {
    this.description = description;
  }

  public EnableFlowSources getEnableFlowSources() {
    return enableFlowSources;
  }

  public void setEnableFlowSources(EnableFlowSources enableFlowSources) {
    this.enableFlowSources = enableFlowSources;
    this.flowSources = getFlowSources(enableFlowSources);

    if (this.munitModule.isPresent() && !this.isIgnore()) {
      this.munitModule.get().addEnableFlowSources(enableFlowSources);
    }
  }

  public String getName() {
    return this.getLocation().getRootContainerName();
  }

  public Event run(Event event) throws Throwable {
    try {
      Event resultingEvent = doExecute(event);

      if (isExpectingFailure()) {
        StringBuilder builder = new StringBuilder();
        builder.append("The test: ").append(getName()).append(" was expecting a failure - ")
            .append("Error ID: ").append(expectedErrorType)
            .append(" - Exception: ").append(expectedException)
            .append(" but it didn't fail");

        fail(builder.toString());
      }
      return resultingEvent;
    } catch (ExecutionException executionException) {
      if (!(executionException.getCause() instanceof ComponentExecutionException)) {
        throw executionException;
      }
      ComponentExecutionException componentExecutionException = (ComponentExecutionException) executionException.getCause();
      Throwable cause = componentExecutionException.getCause();
      Event exceptionEvent = componentExecutionException.getEvent();

      if (isExpectingFailure()) {
        Optional<ComparisonFailure> comparisonFailure = validateExpected(cause, exceptionEvent);
        if (comparisonFailure.isPresent()) {
          throw new AssertionError(comparisonFailure.get().getMessage(), cause);
        } else {
          return exceptionEvent;
        }
      } else {
        throw (cause != null && cause instanceof AssertionError) ? cause : executionException;
      }
    }
  }

  public void startFlowSources() {
    try {
      LifecycleUtils.startIfNeeded(flowSources);
    } catch (MuleException e) {
      throw new MunitError("An error occurred while starting flow sources", e);
    }
  }

  public void stopFlowSources() {
    try {
      LifecycleUtils.stopIfNeeded(flowSources);
    } catch (MuleException e) {
      throw new MunitError("An error occurred while stopping flow sources", e);
    }
  }

  protected Event doExecute(Event event) throws InterruptedException, ExecutionException {
    Either<ExecutionResult, Throwable> submitionResult;
    try {
      submitionResult = scheduler.submit(new ExceptionAwareCallable(event)).get(getTimeout(), TimeUnit.MILLISECONDS);
    } catch (TimeoutException e) {
      throw new MunitError(format("The test '%s' timeout after %s milliseconds", getName(), getTimeout()));
    }

    if (submitionResult.isRight()) {
      Throwable throwable = submitionResult.getRight();
      if (throwable instanceof InterruptedException) {
        throw (InterruptedException) throwable;
      } else if (throwable instanceof ExecutionException) {
        throw (ExecutionException) throwable;
      } else {
        throw new MunitError("Unknown error occurred executing the test", throwable);
      }
    } else {
      ExecutionResult executionResult = submitionResult.getLeft();
      executionResult.complete();
      return executionResult.getEvent();
    }
  }

  protected class ExceptionAwareCallable implements Callable<Either<ExecutionResult, Throwable>> {

    private final Event event;

    ExceptionAwareCallable(Event event) {
      this.event = event;
    }

    @Override
    public Either<ExecutionResult, Throwable> call() throws Exception {
      try {
        return Either.left(execute(InputEvent.create(event)).get());
      } catch (Throwable t) {
        return Either.right(t);
      }
    }
  }

  /**
   * Validates if this Test is expecting to fail
   *
   * @return false, if no errorId nor exception are being expected
   */
  protected boolean isExpectingFailure() {
    return StringUtils.isNotBlank(expectedErrorType) || StringUtils.isNotBlank(expectedException);
  }

  /**
   * Validates if the MUnit Test Flow is expecting this specific exception
   *
   * @param exception the exception
   * @param exceptionEvent the event comming with the exception
   * @return false, is no failure was expected of if the exception does not matches the failure cause expected
   * @throws MunitError in case of major internal error
   */
  protected Optional<ComparisonFailure> validateExpected(Throwable exception, Event exceptionEvent) throws MunitError {
    if (!isExpectingFailure()) {
      return Optional.empty();
    }

    ComparisonFailure failure = null;
    if (StringUtils.isNotBlank(expectedErrorType) && !isErrorIdExpected(exceptionEvent)) {
      String actualErrorType = exceptionEvent.getError().get().getErrorType().toString();
      failure =
          new ComparisonFailure("The error ID thrown does not match the expected one. ", expectedErrorType, actualErrorType);
    } else if (StringUtils.isNotBlank(expectedException) && !isExceptionExpected(exception, exceptionEvent)) {
      String actualException = ofNullable(ExceptionUtils.getRootCause(exception)).orElse(exception).getClass().getName();
      failure =
          new ComparisonFailure("The exception thrown does not match the expected one. ", expectedException, actualException);
    }
    return ofNullable(failure);
  }

  protected boolean isErrorIdExpected(Event exceptionEvent) throws MunitError {
    Object errorId = getExpressionWrapper().evaluateIfExpression(exceptionEvent, expectedErrorType).getValue();
    if (errorId instanceof String) {
      try {
        ComponentIdentifier errorTypeComponentIdentifier =
            ComponentIdentifier.buildFromStringRepresentation(((String) errorId).toUpperCase());
        Optional<ErrorType> errorType = errorTypeRepository.getErrorType(errorTypeComponentIdentifier);
        SingleErrorTypeMatcher errorTypeMatcher = new SingleErrorTypeMatcher(errorType
            .orElseThrow(() -> new MunitError("The expected error type " + errorId + " was not found")));

        Error eventError = exceptionEvent.getError().orElseThrow(() -> new MunitError("The event has no error"));
        return errorTypeMatcher.match(eventError.getErrorType());
      } catch (IllegalStateException e) {
        throw new MunitError("Expect error id " + errorId + " was never registered it can not be thrown", e);
      }
    } else {
      throw new MunitError("Expect error id expression error. The expression should return a valid string");
    }
  }

  protected boolean isExceptionExpected(Throwable exception, Event exceptionEvent) throws MunitError {
    if (getExpressionWrapper().isExpressionValid(expectedException)) {
      return evaluateExpectException(exceptionEvent);
    } else {
      return exceptionMatches(exception);
    }
  }

  protected Boolean evaluateExpectException(Event event) throws MunitError {
    try {
      Object result = getExpressionWrapper().evaluate(event, expectedException).getValue();
      if (result instanceof Boolean) {
        return (Boolean) result;
      } else {
        throw new MunitError("Please make sure your expression matching for expectedException returns a boolean value");
      }
    } catch (ExpressionRuntimeException e) {
      throw new MunitError("Expect exception expression error. " + e.getMessage());
    }
  }

  protected Boolean exceptionMatches(Throwable exception) throws MunitError {
    Class<?> expectExceptionClass;
    ClassLoader appClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      expectExceptionClass = Class.forName(expectedException, true, appClassLoader);
    } catch (ClassNotFoundException e) {
      throw new MunitError("The class " + exception + " could not be found", e);
    }

    return expectExceptionClass
        .isAssignableFrom(ofNullable(ExceptionUtils.getRootCause(exception)).orElse(exception).getClass());
  }

  protected void setErrorTypeRepository(ErrorTypeRepository errorTypeRepository) {
    this.errorTypeRepository = errorTypeRepository;
  }

  protected void setExtendedExpressionManager(ExtendedExpressionManager extendedExpressionManager) {
    this.extendedExpressionManager = extendedExpressionManager;
  }

  protected void setSchedulerService(SchedulerService schedulerService) {
    this.schedulerService = schedulerService;
  }

  private MunitExpressionWrapper getExpressionWrapper() {
    return new MunitExpressionWrapper(extendedExpressionManager);
  }

  private List<MessageSource> getFlowSources(EnableFlowSources enabledFlowSources) {
    return ofNullable(enabledFlowSources)
        .map(flowSources -> flowSources.getFlows().stream().map(EnableFlowSources.FlowRef::getFlow)
            .filter(this::isFlow).map(component -> (Flow) component).map(Pipeline::getSource).collect(Collectors.toList()))
        .orElse(emptyList());
  }

  private boolean isFlow(Component component) {
    if (component instanceof Flow) {
      return true;
    } else {
      logger.warn("Component {} listed in enable-flow-sources section is not a flow", component);
      return false;
    }
  }

  /**
   * @return The current test timeout
   */
  private Integer getTimeout() {
    String property = System.getProperty(MUNIT_TEST_TIMEOUT_PROPERTY);
    if (property != null) {
      return Integer.valueOf(property);
    } else {
      return DEFAULT_TIMEOUT;
    }
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      scheduler.stop();
    }
    super.dispose();
  }

  @Override
  public void initialise() throws InitialisationException {
    scheduler = this.schedulerService.cpuLightScheduler(SchedulerConfig.config().withName("MUnit-Runner"));
    super.initialise();
  }
}
