/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model;

import java.util.concurrent.ExecutionException;

import org.mule.munit.common.event.EventBuilder;
import org.mule.munit.common.util.StackTraceUtil;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.flow.SimpleFlow;
import org.mule.munit.runner.flow.TestFlow;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.core.privileged.exception.EventProcessingException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * MUnit Test. This class is part of the execution model of MUnit.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class Test {

  protected static final transient Logger logger = LoggerFactory.getLogger(Test.class);

  /**
   * The MUnit flow that have to be run before the MUnit test.
   */
  protected BeforeTest before;

  /**
   * The MUnit flow that have to be run after the MUnit test.
   */
  protected AfterTest after;

  /**
   * The MUnit test.
   */
  protected TestFlow test;

  protected MunitModule munitModule;

  public Test(BeforeTest before, TestFlow test, AfterTest after, MunitModule munitModule) {
    this.before = before;
    this.test = test;
    this.after = after;
    this.munitModule = munitModule;
  }

  public String getName() {
    return test.getName();
  }

  public String getDescription() {
    return test.getDescription();
  }

  public boolean isIgnore() {
    return test.isIgnore();
  }

  public TestResult run() {
    TestResult result = new TestResult(getName(), getDescription());
    logger.debug("About to run MUnit test: " + getName());

    if (test.isIgnore()) {
      logger.debug("MUnit test: " + getName() + " is ignored it won't run.");
      result.setSkipped();
      result.setElapsedTime(0);
      return result;
    }

    long testStartTime = System.currentTimeMillis();
    Event event = new EventBuilder(test.getLocation()).withPayload("").build();

    try {
      event = runBefore(event);
      test.startFlowSources();
      event = test.run(event);
    } catch (AssertionError t) {
      result.setFailure(StackTraceUtil.getStackTrace(t));
    } catch (Throwable e) {
      result.setError(StackTraceUtil.getStackTrace(e));
      event = getEventIfPresent(event, e);
    } finally {
      munitModule.reset();
      test.stopFlowSources();
      runAfter(result, event);
    }

    result.setElapsedTime(System.currentTimeMillis() - testStartTime);
    return result;
  }

  private Event runBefore(Event event) throws Throwable {
    logger.debug("Running before test scopes...");
    try {
      return run(event, before);
    } catch (MuleRuntimeException | MuleException e) {
      throw handleRunBeforeFailure(e, e.getCause());
    }
  }

  private void runAfter(TestResult result, Event event) {
    logger.debug("Running after test scopes...");
    try {
      run(event, after);
    } catch (MuleRuntimeException | MuleException e) {
      handleRunAfterFailure(result, e, e.getCause());
    } catch (Throwable e) {
      result.setError(StackTraceUtil.getStackTrace(e));
    }
  }

  private void handleRunAfterFailure(TestResult result, Exception e, Throwable cause) {
    if (cause instanceof AssertionError) {
      result.setFailure(StackTraceUtil.getStackTrace(cause));
    } else {
      result.setError(StackTraceUtil.getStackTrace(e));
    }
  }

  private Throwable handleRunBeforeFailure(Exception e, Throwable cause) throws Throwable {
    if (cause instanceof AssertionError) {
      return cause;
    } else {
      return e;
    }
  }

  protected Event run(Event event, SimpleFlow flow) throws Throwable {
    if (flow != null) {
      try {
        event = flow.execute(event).get();
      } catch (ExecutionException e) {
        throw e.getCause();
      }
    }
    return event;
  }

  private Event getEventIfPresent(Event event, Throwable e) {
    if (e instanceof EventProcessingException) {
      event = ((EventProcessingException) e).getEvent();
    }
    if (e instanceof ComponentExecutionException) {
      event = ((ComponentExecutionException) e).getEvent();
    }
    return event;
  }

}
