/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;

import static org.mule.metadata.java.api.JavaTypeLoader.JAVA;
import static org.mule.runtime.api.meta.Category.COMMUNITY;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.APP_CONFIG;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.munit.common.model.Attachment;
import org.mule.munit.common.model.EventAttributes;
import org.mule.munit.common.model.Payload;
import org.mule.munit.common.model.Property;
import org.mule.munit.common.model.UntypedEventError;
import org.mule.munit.common.model.Variable;
import org.mule.munit.runner.processors.EnableFlowSources;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.declaration.fluent.ConfigurationDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ConstructDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ExtensionDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.OperationDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.OptionalParameterDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.ParameterGroupDeclarer;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModelBuilder;
import org.mule.runtime.extension.api.declaration.type.ExtensionsTypeLoaderFactory;
import org.mule.runtime.extension.api.declaration.type.annotation.LiteralTypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.TypeDslAnnotation;
import org.mule.runtime.extension.api.loader.ExtensionLoadingContext;
import org.mule.runtime.extension.api.loader.ExtensionLoadingDelegate;

/**
 * <p>
 * Provides the MUnit Extension Model for the "crafted" extensionModelLoaderDescriptor
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitExtensionLoadingDelegate implements ExtensionLoadingDelegate {

  // General Attributes
  private static final String VERSION = "2.2.0";
  private static final String DESCRIPTION = "MUnit: Core components";
  private static final String NAME = "MUnit";
  private static final String VENDOR = "MuleSoft, Inc.";
  private static final MuleVersion MIN_MULE_VERSION = new MuleVersion("4.1.0");
  // XML DSL Attributes
  private static final String PREFIX = "munit";
  private static final String NAMESPACE = "http://www.mulesoft.org/schema/mule/munit";
  private static final String SCHEMA_LOCATION = "http://www.mulesoft.org/schema/mule/munit/current/mule-munit.xsd";
  private static final String SCHEMA_VERSION = "2.2.0";
  private static final String XSD_FILE_NAME = "mule-munit.xsd";
  // Configuration Attributes
  private static final String CONFIG_NAME = "config";
  private static final String CONFIG_DESCRIPTION = "MUnit configuration";
  private static final String CONFIG_MIN_MULE_VERSION = "minMuleVersion";
  private static final String CONFIG_MIN_MULE_VERSION_DESCRIPTION = "Minimum Mule Version to run suite against";
  private static final String CONFIG_REQUIRED_PRODUCT = "requiredProduct";
  private static final String CONFIG_REQUIRED_PRODUCT_DESCRIPTION = "Runtime product to run suite against";
  private static final String CONFIG_IGNORE = "ignore";
  private static final String CONFIG_IGNORE_DESCRIPTION = "Ignore suite";
  // Parameterization Attributes
  private static final String PARAMETERIZATIONS_NAME = "parameterizations";
  private static final String PARAMETERIZATIONS_FILE = "file";
  private static final String PARAMETERIZATION_NAME = "parameterization";
  private static final String PARAMETERIZATION_CLASS_NAME = "org.mule.munit.runner.processors.Parameterization";
  private static final String PARAMETERS_NAME = "parameters";
  private static final String PARAMETER_NAME = "parameter";
  private static final String PARAMETER_CLASS_NAME = "org.mule.munit.runner.processors.Parameter";
  private static final String PARAMETERS_PROPERTY_NAME_FIELD = "propertyName";
  private static final String PARAMETERS_VALUE_FIELD = "value";
  private static final String PARAMETERIZATION_NAME_FIELD = "name";
  // Dynamic Port Attributes
  private static final String DYNAMIC_PORT_NAME = "dynamic-port";
  private static final String DYNAMIC_PORT_DESCRIPTION = "Dynamic Port Property";
  private static final String PROPERTY_NAME_NAME = "propertyName";
  private static final String PROPERTY_NAME_DESCRIPTION = "Name of the property for the dynamic port";
  private static final String MIN_NAME = "min";
  private static final String MIN_DESCRIPTION = "Minimum value of the port";
  private static final String MAX_NAME = "max";
  private static final String MAX_DESCRIPTION = "Maximum value of the port";
  // Operations - Test
  private static final String TEST_NAME = "test";
  private static final String TEST_DESCRIPTION = "The MUnit test flow";
  private static final String BEFORE_TEST_NAME = "before-test";
  private static final String BEFORE_TEST_DESCRIPTION = "Flow to be run before each MUnit test, one time per each test";
  private static final String AFTER_TEST_NAME = "after-test";
  private static final String AFTER_TEST_DESCRIPTION = "Flow to be run after each MUnit test, one time per each test";
  private static final String BEFORE_SUITE_NAME = "before-suite";
  private static final String BEFORE_SUITE_DESCRIPTION = "Flow to be run before any other in a MUnit suite, just one time";
  private static final String AFTER_SUITE_NAME = "after-suite";
  private static final String AFTER_SUITE_DESCRIPTION = "Flow to be run after any other in a MUnit suite, just one time";
  private static final String DESCRIPTION_NAME = "description";
  private static final String DESCRIPTION_DESCRIPTION = "Description message to be shown in case of test failure.";
  private static final String IGNORE_NAME = "ignore";
  private static final String IGNORE_DESCRIPTION = "Defines if the test must be ignored.";
  private static final String TAGS_NAME = "tags";
  private static final String TAGS_DESCRIPTION = "Defines the tags for the test.";
  private static final String TIMEOUT_NAME = "timeOut";
  private static final String TIMEOUT_DESCRIPTION = "Defines the maximum running time -in milliseconds- for the test.";
  private static final String EXPECTED_ERROR_TYPE_NAME = "expectedErrorType";
  private static final String EXPECTED_ERROR_TYPE_DESCRIPTION = "Error ID expected on this test.";
  private static final String EXPECTED_EXCEPTION_NAME = "expectedException";
  private static final String EXPECTED_EXCEPTION_DESCRIPTION = "Error expected on this test.";
  private static final String EXPECTED_ERROR_DESCRIPTION_NAME = "expectedErrorDescription";
  private static final String EXPECTED_ERROR_DESCRIPTION_DESCRIPTION = "Error description expected on this test.";
  private static final String ENABLE_FLOW_SOURCES_NAME = "enableFlowSources";
  private static final String ENABLE_FLOW_SOURCES_DESCRIPTION = "Flow sources to enable during the test";
  private static final String BEHAVIOR_NAME = "behavior";
  private static final String BEHAVIOR_DESCRIPTION = "Processors that define test behavior";
  private static final String EXECUTION_NAME = "execution";
  private static final String EXECUTION_DESCRIPTION = "Processors that define execution of the test";
  private static final String VALIDATION_NAME = "validation";
  private static final String VALIDATION_DESCRIPTION = "Processors that define validation of the test";
  // Operations - Set Event
  private static final String SET_EVENT_NAME = "set-event";
  private static final String SET_EVENT_DESCRIPTION = "Defines the event to be used for testing";
  private static final String CLONE_ORIGINAL_EVENT_NAME = "cloneOriginalEvent";
  private static final String CLONE_ORIGINAL_EVENT_DESCRIPTION = "Define if the original Event should be cloned";
  private static final String PAYLOAD_NAME = "payload";
  private static final String PAYLOAD_DESCRIPTION = "The payload to be set.";
  private static final String ATTRIBUTES_NAME = "attributes";
  private static final String ATTRIBUTES_DESCRIPTION = "Attributes to be set.";
  private static final String ERROR_NAME = "error";
  private static final String ERROR_DESCRIPTION = "Error to be set.";
  private static final String VARIABLE_NAME = "variables";
  private static final String VARIABLE_DESCRIPTION = "Variables to be set.";
  private static final String INBOUND_PROPERTIES_NAME = "inbound-properties";
  private static final String INBOUND_PROPERTIES_DESCRIPTION = "Inbound properties to be set.";
  private static final String OUTBOUND_PROPERTIES_NAME = "outbound-properties";
  private static final String OUTBOUND_PROPERTIES_DESCRIPTION = "Outbound properties to be set.";
  private static final String INBOUND_ATTACHMENTS_NAME = "inbound-attachments";
  private static final String INBOUND_ATTACHMENTS_DESCRIPTION = "Inbound attachments to be set.";
  private static final String OUTBOUND_ATTACHMENTS_NAME = "outbound-attachments";
  private static final String OUTBOUND_ATTACHMENTS_DESCRIPTION = "Outbound attachments to be set.";
  // Operations - Set Null Payload
  private static final String SET_NULL_PAYLOAD_NAME = "set-null-payload";
  private static final String SET_NULL_PAYLOAD_DESCRIPTION = "Defines a Null payload for testing";

  private ClassTypeLoader typeLoader = ExtensionsTypeLoaderFactory.getDefault().createTypeLoader();
  private BaseTypeBuilder typeBuilder = BaseTypeBuilder.create(JAVA);

  @Override
  public void accept(ExtensionDeclarer extensionDeclarer, ExtensionLoadingContext extensionLoadingContext) {

    declareGeneral(extensionDeclarer);
    declareXmlDsl(extensionDeclarer);
    declareConfiguration(extensionDeclarer);
    declareOperations(extensionDeclarer);
    declareDynamicPort(extensionDeclarer);
  }

  private void declareGeneral(ExtensionDeclarer extensionDeclarer) {
    extensionDeclarer
        .named(NAME)
        .describedAs(DESCRIPTION)
        .onVersion(VERSION)
        .fromVendor(VENDOR)
        .withCategory(COMMUNITY);
  }

  private void declareXmlDsl(ExtensionDeclarer extensionDeclarer) {
    XmlDslModel xmlDslModel = XmlDslModel.builder()
        .setPrefix(PREFIX)
        .setNamespace(NAMESPACE)
        .setSchemaLocation(SCHEMA_LOCATION)
        .setSchemaVersion(SCHEMA_VERSION)
        .setXsdFileName(XSD_FILE_NAME).build();
    extensionDeclarer.withXmlDsl(xmlDslModel);
  }

  private void declareConfiguration(ExtensionDeclarer extensionDeclarer) {

    ConfigurationDeclarer configurationDeclarer = extensionDeclarer
        .withConfig(CONFIG_NAME)
        .describedAs(CONFIG_DESCRIPTION);

    ParameterGroupDeclarer parameterGroupDeclarer = configurationDeclarer.onDefaultParameterGroup();
    parameterGroupDeclarer.withOptionalParameter(CONFIG_MIN_MULE_VERSION).describedAs(CONFIG_MIN_MULE_VERSION_DESCRIPTION)
        .ofType(typeBuilder.stringType().build()).withExpressionSupport(NOT_SUPPORTED);
    parameterGroupDeclarer.withOptionalParameter(CONFIG_REQUIRED_PRODUCT).describedAs(CONFIG_REQUIRED_PRODUCT_DESCRIPTION)
        .ofType(typeBuilder.stringType().build()).withExpressionSupport(NOT_SUPPORTED);
    parameterGroupDeclarer.withOptionalParameter(CONFIG_IGNORE).describedAs(CONFIG_IGNORE_DESCRIPTION)
        .ofType(typeBuilder.booleanType().build()).withExpressionSupport(NOT_SUPPORTED);
    declareParameterizations(parameterGroupDeclarer);
  }

  private void declareParameterizations(ParameterGroupDeclarer parameterGroupDeclarer) {
    ObjectTypeBuilder parameterType = typeBuilder.objectType()
        .with(new TypeAliasAnnotation(PARAMETER_NAME))
        .with(new TypeIdAnnotation(PARAMETER_CLASS_NAME))
        .with(new TypeDslAnnotation(true, false, null, null));

    parameterType.addField().key(PARAMETERS_PROPERTY_NAME_FIELD).value().stringType()
        .with(new LiteralTypeAnnotation());
    parameterType.addField().key(PARAMETERS_VALUE_FIELD).value().stringType()
        .with(new LiteralTypeAnnotation());

    ObjectTypeBuilder parameterizationType = typeBuilder.objectType()
        .with(new TypeAliasAnnotation(PARAMETERIZATION_NAME))
        .with(new TypeIdAnnotation(PARAMETERIZATION_CLASS_NAME))
        .with(new TypeDslAnnotation(true, false, null, null));
    parameterizationType.addField().key(PARAMETERIZATION_NAME_FIELD).required().value().stringType()
        .with(new LiteralTypeAnnotation());
    parameterizationType.addField().key(PARAMETERS_NAME).value(typeBuilder.arrayType().of(parameterType))
        .with(new LiteralTypeAnnotation());

    parameterGroupDeclarer
        .withOptionalParameter(PARAMETERIZATIONS_NAME)
        .ofType(typeBuilder.arrayType().of(parameterizationType).build())
        .withExpressionSupport(NOT_SUPPORTED)
        .withDsl(ParameterDslConfiguration.builder()
            .allowsInlineDefinition(true)
            .allowTopLevelDefinition(false)
            .allowsReferences(false)
            .build());

    parameterGroupDeclarer.withOptionalParameter(PARAMETERIZATIONS_FILE)
        .ofType(typeBuilder.stringType().build()).withExpressionSupport(NOT_SUPPORTED);
  }

  private void declareDynamicPort(ExtensionDeclarer extensionDeclarer) {
    ConstructDeclarer configuration = extensionDeclarer.withConstruct(DYNAMIC_PORT_NAME)
        .allowingTopLevelDefinition()
        .withStereotype(APP_CONFIG)
        .describedAs(DYNAMIC_PORT_DESCRIPTION);

    configuration.onDefaultParameterGroup()
        .withRequiredParameter(PROPERTY_NAME_NAME)
        .ofType(typeLoader.load(String.class))
        .withExpressionSupport(NOT_SUPPORTED)
        .describedAs(PROPERTY_NAME_DESCRIPTION);

    configuration.onDefaultParameterGroup()
        .withOptionalParameter(MIN_NAME)
        .ofType(typeLoader.load(Integer.class))
        .withExpressionSupport(NOT_SUPPORTED)
        .describedAs(MIN_DESCRIPTION);

    configuration.onDefaultParameterGroup()
        .withOptionalParameter(MAX_NAME)
        .ofType(typeLoader.load(Integer.class))
        .withExpressionSupport(NOT_SUPPORTED)
        .describedAs(MAX_DESCRIPTION);
  }

  private void declareOperations(ExtensionDeclarer extensionDeclarer) {
    declareMunitTestFlow(extensionDeclarer);
    declareMunitFlow(extensionDeclarer, BEFORE_TEST_NAME, BEFORE_TEST_DESCRIPTION);
    declareMunitFlow(extensionDeclarer, AFTER_TEST_NAME, AFTER_TEST_DESCRIPTION);
    declareMunitFlow(extensionDeclarer, BEFORE_SUITE_NAME, BEFORE_SUITE_DESCRIPTION);
    declareMunitFlow(extensionDeclarer, AFTER_SUITE_NAME, AFTER_SUITE_DESCRIPTION);
    declareSetEvent(extensionDeclarer);
    declareSetNullPayload(extensionDeclarer);
  }

  private void declareMunitTestFlow(ExtensionDeclarer extensionDeclarer) {
    ConstructDeclarer testDeclarer = declareMunitFlow(extensionDeclarer, TEST_NAME, TEST_DESCRIPTION);
    ParameterGroupDeclarer parameterGroupDeclarer = testDeclarer.onDefaultParameterGroup();
    addOptionalParameter(IGNORE_NAME, IGNORE_DESCRIPTION, Boolean.class, parameterGroupDeclarer);
    addOptionalParameter(TAGS_NAME, TAGS_DESCRIPTION, String.class, parameterGroupDeclarer);
    addOptionalParameter(TIMEOUT_NAME, TIMEOUT_DESCRIPTION, Integer.class, parameterGroupDeclarer);
    addOptionalParameter(EXPECTED_ERROR_TYPE_NAME, EXPECTED_ERROR_TYPE_DESCRIPTION, String.class, parameterGroupDeclarer);
    addOptionalParameter(EXPECTED_EXCEPTION_NAME, EXPECTED_EXCEPTION_DESCRIPTION, String.class, parameterGroupDeclarer);
    addOptionalParameter(EXPECTED_ERROR_DESCRIPTION_NAME, EXPECTED_ERROR_DESCRIPTION_DESCRIPTION, String.class,
                         parameterGroupDeclarer);

    declareEnableFlowSources(parameterGroupDeclarer);
    declareMunitTestSection(BEHAVIOR_NAME, BEHAVIOR_DESCRIPTION, extensionDeclarer, testDeclarer);
    declareMunitTestSection(EXECUTION_NAME, EXECUTION_DESCRIPTION, extensionDeclarer, testDeclarer);
    declareMunitTestSection(VALIDATION_NAME, VALIDATION_DESCRIPTION, extensionDeclarer, testDeclarer);
  }

  private void declareEnableFlowSources(ParameterGroupDeclarer parameterGroupDeclarer) {
    parameterGroupDeclarer
        .withOptionalParameter(ENABLE_FLOW_SOURCES_NAME)
        .describedAs(ENABLE_FLOW_SOURCES_DESCRIPTION)
        .ofType(typeLoader.load(EnableFlowSources.class))
        .withExpressionSupport(NOT_SUPPORTED);
  }

  private void declareMunitTestSection(String name, String description, ExtensionDeclarer extensionDeclarer,
                                       ConstructDeclarer testDeclarer) {
    StereotypeModel stereoType = StereotypeModelBuilder.newStereotype(name, PREFIX).build();
    extensionDeclarer.withConstruct(name).describedAs(description).withStereotype(stereoType).withChain();
    testDeclarer.withOptionalComponent(name).withAllowedStereotypes(stereoType);
  }

  private ConstructDeclarer declareMunitFlow(ExtensionDeclarer extensionDeclarer, String name, String description) {
    ConstructDeclarer constructDeclarer = extensionDeclarer.withConstruct(name)
        .describedAs(description)
        .allowingTopLevelDefinition();

    addOptionalParameter(DESCRIPTION_NAME, DESCRIPTION_DESCRIPTION, String.class, constructDeclarer.onDefaultParameterGroup());
    return constructDeclarer;
  }

  private void declareSetEvent(ExtensionDeclarer extensionDeclarer) {
    OperationDeclarer operationDeclarer = extensionDeclarer
        .withOperation(SET_EVENT_NAME)
        .describedAs(SET_EVENT_DESCRIPTION);
    operationDeclarer.withOutputAttributes().ofType(typeLoader.load(Object.class));
    operationDeclarer.withOutput().ofType(typeBuilder.anyType().build());

    ParameterGroupDeclarer declarer = operationDeclarer.onDefaultParameterGroup();
    addOptionalParameter(CLONE_ORIGINAL_EVENT_NAME, CLONE_ORIGINAL_EVENT_DESCRIPTION, Boolean.class, declarer);
    addOptionalParameter(PAYLOAD_NAME, PAYLOAD_DESCRIPTION, Payload.class, declarer);
    addOptionalParameter(ATTRIBUTES_NAME, ATTRIBUTES_DESCRIPTION, EventAttributes.class, declarer);
    addOptionalParameter(ERROR_NAME, ERROR_DESCRIPTION, UntypedEventError.class, declarer);
    addOptionalListParameter(VARIABLE_NAME, VARIABLE_DESCRIPTION, Variable.class, declarer);
    addOptionalListParameter(INBOUND_PROPERTIES_NAME, INBOUND_PROPERTIES_DESCRIPTION, Property.class, declarer);
    addOptionalListParameter(OUTBOUND_PROPERTIES_NAME, OUTBOUND_PROPERTIES_DESCRIPTION, Property.class, declarer);
    addOptionalListParameter(INBOUND_ATTACHMENTS_NAME, INBOUND_ATTACHMENTS_DESCRIPTION, Attachment.class, declarer);
    addOptionalListParameter(OUTBOUND_ATTACHMENTS_NAME, OUTBOUND_ATTACHMENTS_DESCRIPTION, Attachment.class, declarer);
  }

  private void declareSetNullPayload(ExtensionDeclarer extensionDeclarer) {
    OperationDeclarer operationDeclarer = extensionDeclarer
        .withOperation(SET_NULL_PAYLOAD_NAME)
        .describedAs(SET_NULL_PAYLOAD_DESCRIPTION);
    operationDeclarer.withOutputAttributes().ofType(typeLoader.load(Object.class));
    operationDeclarer.withOutput().ofType(typeBuilder.anyType().build());
  }

  private OptionalParameterDeclarer addOptionalParameter(String name, String description, Class<?> clazz,
                                                         ParameterGroupDeclarer declarer) {
    return addOptionalParameter(name, description, typeLoader.load(clazz), declarer);
  }

  private OptionalParameterDeclarer addOptionalListParameter(String name, String description, Class<?> clazz,
                                                             ParameterGroupDeclarer declarer) {
    return addOptionalParameter(name, description, typeBuilder.arrayType().of(typeLoader.load(clazz)).build(), declarer);
  }

  private OptionalParameterDeclarer addOptionalParameter(String name, String description, MetadataType type,
                                                         ParameterGroupDeclarer declarer) {
    return declarer
        .withOptionalParameter(name)
        .describedAs(description)
        .ofType(type);
  }

}
