/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model;

import static java.util.Optional.ofNullable;

import java.util.concurrent.ExecutionException;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.mule.munit.common.event.EventBuilder;
import org.mule.munit.common.util.StackTraceUtil;
import org.mule.munit.runner.component.TestComponent;
import org.mule.munit.runner.component.rules.TestDescription;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.flow.SimpleFlow;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.core.privileged.exception.EventProcessingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * MUnit Test. This class is part of the execution model of MUnit.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class Test {

  protected static final transient Logger logger = LoggerFactory.getLogger(Test.class);

  /**
   * The MUnit flow that have to be run before the MUnit test.
   */
  protected BeforeTest before;

  /**
   * The MUnit flow that have to be run after the MUnit test.
   */
  protected AfterTest after;

  /**
   * The MUnit test.
   */
  protected TestComponent test;

  protected MunitModule munitModule;

  public Test(BeforeTest before, TestComponent test, AfterTest after, MunitModule munitModule) {
    this.before = before;
    this.test = test;
    this.after = after;
    this.munitModule = munitModule;
  }

  public String getName() {
    return test.getName();
  }

  public String getDescription() {
    return test.getDescription();
  }

  public boolean isIgnore() {
    return test.isIgnored();
  }

  public TestResult run() {
    TestResult result = new TestResult(getName(), getDescription());
    logger.debug("About to run MUnit test: " + getName());

    if (test.isIgnored()) {
      logger.debug("MUnit test: " + getName() + " is ignored it won't run.");
      result.setSkipped();
      result.setElapsedTime(0);
      return result;
    }

    long testStartTime = System.currentTimeMillis();
    Event event = new EventBuilder(test.getLocation()).withPayload("").build();

    try {
      munitModule.applyRules(createTestDescription());
      event = runBefore(event);
      test.setUp();
      event = test.run(event);
    } catch (Throwable e) {
      handleTestFailure(result, getCause(e));
      event = getEventIfPresent(event, e);
    } finally {
      runFinally(result, () -> test.tearDown());
      runAfter(result, event);
      runFinally(result, () -> munitModule.reset());
    }

    result.setElapsedTime(System.currentTimeMillis() - testStartTime);
    return result;
  }

  private Event runBefore(Event event) throws Throwable {
    logger.debug("Running before test scopes...");
    try {
      return run(event, before);
    } catch (MuleRuntimeException | MuleException e) {
      throw getCause(e);
    }
  }

  private void runAfter(TestResult result, Event event) {
    logger.debug("Running after test scopes...");
    try {
      run(event, after);
    } catch (Throwable e) {
      handleTestFailure(result, getCause(e));
    }
  }

  private void handleTestFailure(TestResult result, Throwable cause) {
    if (cause instanceof AssertionError) {
      result.setFailure(StackTraceUtil.getStackTrace(cause));
    } else {
      result.setError(StackTraceUtil.getStackTrace(cause));
    }
  }

  protected Event run(Event event, SimpleFlow flow) throws Throwable {
    if (flow != null) {
      try {
        event = flow.execute(event).get();
      } catch (ExecutionException e) {
        throw e.getCause();
      }
    }
    return event;
  }

  private Event getEventIfPresent(Event event, Throwable e) {
    if (e instanceof EventProcessingException) {
      event = ((EventProcessingException) e).getEvent();
    }
    if (e instanceof ComponentExecutionException) {
      event = ((ComponentExecutionException) e).getEvent();
    }
    return event;
  }

  private TestDescription createTestDescription() {
    return TestDescription.builder().name(getName()).description(getDescription()).ignored(isIgnore()).tags(test.getTags())
        .build();
  }

  private Throwable getCause(Throwable e) {
    return e instanceof AssertionError ? e : ofNullable(ExceptionUtils.getRootCause(e)).orElse(e);
  }

  private void runFinally(TestResult result, Runnable action) {
    try {
      action.run();
    } catch (Throwable e) {
      handleTestFailure(result, getCause(e));
    }
  }

}
