/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model.builders;

import org.mule.munit.runner.component.TestComponent;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

/**
 * Given a {@Link MunitTestFlow} it validates if the test should be run, based of a list of names and a set of tags
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class TestRunFilter {

  public static final String NO_TAG_TOKEN = "NO_TAG";

  /**
   * Define if a test should be run
   * 
   * @param testComponent the test to be checked if it should be run
   * @param testNameList a list of test names
   * @param tags a list of tags
   * @return false if the test should not be run, true otherwise
   */
  public boolean shouldRunTest(TestComponent testComponent, List<String> testNameList, Set<String> tags) {
    return isTestInList(testComponent, testNameList) && isTagInTest(testComponent, tags);
  }

  private boolean isTestInList(TestComponent testComponent, List<String> testNameList) {
    if (null == testNameList || testNameList.isEmpty()) {
      return true;
    }
    return testNameList.stream().filter(StringUtils::isNotBlank).anyMatch(name -> testComponent.getName().matches(name));
  }

  private boolean isTagInTest(TestComponent testFlow, Set<String> tags) {
    return tags == null || tags.isEmpty() || shouldRunWithNoTag(testFlow, tags)
        || tagsInCommon(testFlow.getTags(), tags);
  }

  private boolean tagsInCommon(Set<String> testTags, Set<String> userTags) {
    return !Collections.disjoint(toLowerCaseAndTrim(testTags), toLowerCaseAndTrim(userTags));
  }

  private Set<String> toLowerCaseAndTrim(Set<String> tags) {
    return tags.stream().map(StringUtils::lowerCase).map(String::trim).collect(Collectors.toSet());
  }

  private boolean shouldRunWithNoTag(TestComponent testFlow, Set<String> tags) {
    return testFlow.getTags().isEmpty() && tags.stream().anyMatch(tag -> tag.trim().equalsIgnoreCase(NO_TAG_TOKEN));
  }


}
