/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.flow;

import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static junit.framework.Assert.fail;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.exception.ExceptionUtils.getRootCause;
import static org.mule.munit.runner.model.builders.TestRunFilter.NO_TAG_TOKEN;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.junit.ComparisonFailure;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.util.MunitExpressionWrapper;
import org.mule.munit.runner.component.TestComponent;
import org.mule.munit.runner.model.TestExecutionException;
import org.mule.munit.runner.processors.EnableFlowSources;
import org.mule.munit.runner.processors.EnableFlowSources.FlowRef;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.execution.ComponentExecutionException;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerConfig;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.api.construct.Pipeline;
import org.mule.runtime.core.api.el.ExtendedExpressionManager;
import org.mule.runtime.core.api.exception.SingleErrorTypeMatcher;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.core.api.functional.Either;
import org.mule.runtime.core.api.lifecycle.LifecycleUtils;
import org.mule.runtime.core.api.processor.ReactiveProcessor;
import org.mule.runtime.core.api.processor.strategy.ProcessingStrategy;
import org.mule.runtime.core.api.source.MessageSource;
import org.mule.runtime.core.privileged.processor.CompositeProcessorChainRouter;
import org.mule.runtime.core.privileged.processor.chain.MessageProcessorChain;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * The Test Flow
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class TestFlow extends CompositeProcessorChainRouter implements TestComponent {

  public static final String MUNIT_TEST_TIMEOUT_PROPERTY = "munit.test.timeout";

  protected static final transient Logger logger = LoggerFactory.getLogger(TestFlow.class);

  private static final int DEFAULT_TIMEOUT = 120000;

  @Inject
  private ErrorTypeRepository errorTypeRepository;

  @Inject
  private ExtendedExpressionManager extendedExpressionManager;

  @Inject
  protected Optional<MunitModule> munitModule;

  @Inject
  private SchedulerService schedulerService;

  private static final String TAG_SEPARATOR = ",";

  /**
   * <p>
   * The description of the test
   * </p>
   */
  private String description;

  /**
   * <p>
   * Determines if the test has to be ignored
   * </p>
   */
  private String ignore;

  /**
   * <p>
   * The error type id that is expected
   * </p>
   */
  private String expectedErrorType;

  /**
   * <p>
   * The name of the exception that is expected
   * </p>
   */
  private String expectedException;

  /**
   * <p>
   * The description of the error that is expected
   * </p>
   */
  private String expectedErrorDescription;

  /**
   * <p>
   * The tags that the test has
   * </p>
   */
  private String tags;

  /**
   * <p>
   * List the flow whose sources should be enabled
   * </p>
   */
  private List<FlowRef> enableFlowSources;

  /**
   * <p>
   * List the sources which should be enabled
   * </p>
   */
  private List<MessageSource> flowSources = emptyList();

  private Integer timeOut;

  private Scheduler scheduler;

  /**
   * @throws InitialisationException
   */
  private Map<String, ProcessingStrategy> childProcessingStrategies = new HashMap<>();

  private final Pattern booleanPattern = Pattern.compile("true|false");

  @Override
  public boolean isIgnored() {
    validateIgnore(ignore);

    return booleanPattern.matcher(ignore).matches() ? Boolean.valueOf(ignore)
        : (boolean) getExpressionWrapper().evaluate(ignore).getValue();
  }

  public void setIgnore(String ignore) {
    validateIgnore(ignore);
    this.ignore = ignore;
  }

  public String getExpectedErrorType() {
    return expectedErrorType;
  }

  public void setExpectedErrorType(String expectedErrorType) {
    this.expectedErrorType = expectedErrorType;
  }

  public String getExpectedException() {
    return expectedException;
  }

  public void setExpectedException(String expectedException) {
    this.expectedException = expectedException;
  }

  public String getExpectedErrorDescription() {
    return expectedErrorDescription;
  }

  public void setExpectedErrorDescription(String expectedErrorDescription) {
    this.expectedErrorDescription = expectedErrorDescription;
  }

  public void setTimeOut(Integer timeout) {
    this.timeOut = timeout;
  }

  public Integer getTimeOut() {
    return timeOut;
  }

  @Override
  public Set<String> getTags() {
    if (isBlank(tags)) {
      return Collections.emptySet();
    }
    Set<String> tagSet = Stream.of(tags.split(TAG_SEPARATOR)).collect(Collectors.toSet());
    if (tagSet.stream().anyMatch(tag -> tag.trim().equalsIgnoreCase(NO_TAG_TOKEN))) {
      throw new IllegalArgumentException("The tag '" + NO_TAG_TOKEN + "' is invalid since it's a keyword.");
    }
    return tagSet;
  }

  public void setTags(String tags) {
    this.tags = tags;
  }

  @Override
  public String getDescription() {
    return description == null ? StringUtils.EMPTY : description;
  }

  public void setDescription(String description) {
    this.description = description;
  }

  public List<FlowRef> getEnableFlowSources() {
    return enableFlowSources;
  }

  public void setEnableFlowSources(List<FlowRef> enableFlowSources) {
    this.enableFlowSources = enableFlowSources;

    this.flowSources = getFlowSources(enableFlowSources);

    if (this.munitModule.isPresent() && !this.isIgnored()) {
      this.munitModule.get().addEnableFlowSources(enableFlowSources);
    }
  }

  @Override
  public String getName() {
    return this.getLocation().getRootContainerName();
  }

  @Override
  public Event run(Event event) throws Throwable {
    try {
      Event resultingEvent = doExecute(event);

      if (isExpectingFailure()) {
        StringBuilder builder = new StringBuilder();
        builder.append("The test: ").append(getName()).append(" was expecting a failure");

        if (expectedErrorType != null) {
          builder.append(" - Error ID: ").append(expectedErrorType);
        }

        if (expectedException != null) {
          builder.append(" - Exception: ").append(expectedException);
        }

        if (expectedErrorDescription != null) {
          builder.append(" - Error Description: ").append(expectedErrorDescription);
        }

        builder.append(" but it didn't fail");

        fail(builder.toString());
      }
      return resultingEvent;
    } catch (ExecutionException executionException) {
      if (!(executionException.getCause() instanceof ComponentExecutionException)) {
        throw executionException;
      }
      ComponentExecutionException componentExecutionException = (ComponentExecutionException) executionException.getCause();
      Event exceptionEvent = componentExecutionException.getEvent();
      Error error = exceptionEvent.getError()
          .orElseThrow(() -> new MunitError("Test " + getName() + " failed but no error is present in event"));

      if (isExpectingFailure()) {
        Optional<ComparisonFailure> comparisonFailure = validateExpected(error, exceptionEvent);
        if (comparisonFailure.isPresent()) {
          throw new AssertionError(comparisonFailure.get().getMessage(), error.getCause());
        } else {
          return exceptionEvent;
        }
      } else {
        Throwable rootCause = ofNullable(getRootCause(error.getCause())).orElse(error.getCause());
        throw (rootCause instanceof AssertionError) ? rootCause
            : new TestExecutionException(executionException, error, getName());
      }
    }
  }

  @Override
  public void setUp() {
    startFlowSources();
  }

  @Override
  public void tearDown() {
    stopFlowSources();
  }

  public void startFlowSources() {
    try {
      LifecycleUtils.startIfNeeded(flowSources);
    } catch (MuleException e) {
      throw new MunitError("An error occurred while starting flow sources", e);
    }
  }

  public void stopFlowSources() {
    try {
      LifecycleUtils.stopIfNeeded(flowSources);
    } catch (MuleException e) {
      throw new MunitError("An error occurred while stopping flow sources", e);
    }
  }

  protected Event doExecute(Event event) throws InterruptedException, ExecutionException {
    Either<Event, Throwable> submissionResult;
    try {
      submissionResult = scheduler.submit(new ExceptionAwareCallable(event)).get(getTimeout(), TimeUnit.MILLISECONDS);
    } catch (TimeoutException e) {
      throw new MunitError(format("The test '%s' timeout after %s milliseconds", getName(), getTimeout()));
    }

    if (submissionResult.isRight()) {
      Throwable throwable = submissionResult.getRight();
      if (throwable instanceof InterruptedException) {
        throw (InterruptedException) throwable;
      } else if (throwable instanceof ExecutionException) {
        throw (ExecutionException) throwable;
      } else {
        throw new MunitError("Unknown error occurred executing the test", throwable);
      }
    } else {
      return submissionResult.getLeft();
    }
  }

  protected class ExceptionAwareCallable implements Callable<Either<Event, Throwable>> {

    private final Event event;

    ExceptionAwareCallable(Event event) {
      this.event = event;
    }

    @Override
    public Either<Event, Throwable> call() {
      try {
        return Either.left(execute(event).get());
      } catch (Throwable t) {
        return Either.right(t);
      }
    }
  }

  /**
   * Validates if this Test is expecting to fail
   *
   * @return false, if no errorId nor exception are being expected
   */
  protected boolean isExpectingFailure() {
    return StringUtils.isNotBlank(expectedErrorType) || StringUtils.isNotBlank(expectedException)
        || StringUtils.isNotBlank(expectedErrorDescription);
  }

  /**
   * Validates if the MUnit Test Flow is expecting this specific exception
   *
   * @param error          the error inside the {@link Event}
   * @param exceptionEvent the event coming with the exception
   * @return false, is no failure was expected of if the exception does not matches the failure cause expected
   * @throws MunitError in case of major internal error
   */
  protected Optional<ComparisonFailure> validateExpected(Error error, Event exceptionEvent) throws MunitError {
    if (!isExpectingFailure()) {
      return Optional.empty();
    }

    Throwable cause = error.getCause();
    ComparisonFailure failure = null;
    if (StringUtils.isNotBlank(expectedErrorType) && !isErrorIdExpected(exceptionEvent)) {
      String actualErrorType = error.getErrorType().toString();
      failure =
          new ComparisonFailure("The error ID thrown does not match the expected one. ", expectedErrorType, actualErrorType);
    } else if (StringUtils.isNotBlank(expectedException) && !isExceptionExpected(cause, exceptionEvent)) {
      String actualException = cause.getClass().getName();
      failure =
          new ComparisonFailure("The exception thrown does not match the expected one. ", expectedException, actualException);
    } else if (StringUtils.isNotBlank(expectedErrorDescription) && !isErrorDescriptionExpected(error)) {
      String actualErrorDescription = error.getDescription();
      failure =
          new ComparisonFailure("The error description thrown does not match the expected one.", expectedErrorDescription,
                                actualErrorDescription);
    }

    return ofNullable(failure);
  }

  protected boolean isErrorIdExpected(Event exceptionEvent) throws MunitError {
    Object errorId = getExpressionWrapper().evaluateIfExpression(exceptionEvent, expectedErrorType).getValue();
    if (errorId instanceof String) {
      try {
        ComponentIdentifier errorTypeComponentIdentifier =
            ComponentIdentifier.buildFromStringRepresentation(((String) errorId).toUpperCase());
        Optional<ErrorType> errorType = errorTypeRepository.getErrorType(errorTypeComponentIdentifier);
        SingleErrorTypeMatcher errorTypeMatcher = new SingleErrorTypeMatcher(errorType
            .orElseThrow(() -> new MunitError("The expected error type " + errorId + " was not found")));

        Error eventError = exceptionEvent.getError().orElseThrow(() -> new MunitError("The event has no error"));
        return errorTypeMatcher.match(eventError.getErrorType());
      } catch (IllegalStateException e) {
        throw new MunitError("Expect error id " + errorId + " was never registered it can not be thrown", e);
      }
    } else {
      throw new MunitError("Expect error id expression error. The expression should return a valid string");
    }
  }

  protected boolean isExceptionExpected(Throwable exception, Event exceptionEvent) throws MunitError {
    if (getExpressionWrapper().isExpressionValid(expectedException)) {
      return evaluateExpectException(exceptionEvent);
    } else {
      return exceptionMatches(exception);
    }
  }

  protected boolean isErrorDescriptionExpected(Error error) throws MunitError {
    return error.getDescription().contains(expectedErrorDescription);
  }

  protected Boolean evaluateExpectException(Event event) throws MunitError {
    try {
      Object result = getExpressionWrapper().evaluate(event, expectedException).getValue();
      if (result instanceof Boolean) {
        return (Boolean) result;
      } else {
        throw new MunitError("Please make sure your expression matching for expectedException returns a boolean value");
      }
    } catch (ExpressionRuntimeException e) {
      throw new MunitError("Expect exception expression error. " + e.getMessage());
    }
  }

  protected Boolean exceptionMatches(Throwable exception) throws MunitError {
    Class<?> expectExceptionClass;
    ClassLoader appClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      expectExceptionClass = Class.forName(expectedException, true, appClassLoader);
    } catch (ClassNotFoundException e) {
      throw new MunitError("The class " + exception + " could not be found", e);
    }

    return expectExceptionClass.isAssignableFrom(exception.getClass());
  }

  protected void setErrorTypeRepository(ErrorTypeRepository errorTypeRepository) {
    this.errorTypeRepository = errorTypeRepository;
  }

  protected void setExtendedExpressionManager(ExtendedExpressionManager extendedExpressionManager) {
    this.extendedExpressionManager = extendedExpressionManager;
  }

  protected void setSchedulerService(SchedulerService schedulerService) {
    this.schedulerService = schedulerService;
  }

  public void addProcessingStrategy(String childName, ProcessingStrategy processingStrategy) {
    this.childProcessingStrategies.put(childName, processingStrategy);
  }

  @Override
  public void initialise() throws InitialisationException {
    scheduler = this.schedulerService.cpuLightScheduler(SchedulerConfig.config().withName("MUnit-Runner"));
    super.initialise();
  }

  @Override
  public void start() throws MuleException {
    LifecycleUtils.startIfNeeded(childProcessingStrategies.values());
    super.start();
  }

  @Override
  public void stop() throws MuleException {
    super.stop();
    LifecycleUtils.stopIfNeeded(childProcessingStrategies.values());
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      scheduler.stop();
    }
    if (childProcessingStrategies != null) {
      LifecycleUtils.disposeIfNeeded(childProcessingStrategies.values(), logger);
    }
    super.dispose();
  }

  /**
   * We override this method that is present since Mule Version 4.2.0. When MUnit has a minMuleVersion 4.2.0, we can remove the
   * reflection from this method and use the proper override annotation.
   */
  protected List<ReactiveProcessor> processorChainsToExecute(List<MessageProcessorChain> chains) {
    return chains.stream().map(chain -> lookupProcessingStrategy(chain).map(strategy -> strategy.onPipeline(chain)).orElse(null))
        .filter(Objects::nonNull).collect(Collectors.toList());
  }

  private Optional<ProcessingStrategy> lookupProcessingStrategy(MessageProcessorChain chain) {
    return ofNullable(childProcessingStrategies.get(chain.getLocation().getComponentIdentifier().getIdentifier().getName()));
  }

  private MunitExpressionWrapper getExpressionWrapper() {
    return new MunitExpressionWrapper(extendedExpressionManager);
  }

  private List<MessageSource> getFlowSources(List<FlowRef> enabledFlowSources) {
    return ofNullable(enabledFlowSources)
        .map(flowSources -> flowSources.stream().map(FlowRef::getFlow)
            .filter(this::isFlow).map(component -> (Flow) component).map(Pipeline::getSource).collect(toList()))
        .orElse(emptyList());
  }

  private boolean isFlow(Component component) {
    if (component instanceof Flow) {
      return true;
    } else {
      logger.warn("Component {} listed in enable-flow-sources section is not a flow", component);
      return false;
    }
  }

  /**
   * @return The current test timeout
   */
  private Integer getTimeout() {
    String property = System.getProperty(MUNIT_TEST_TIMEOUT_PROPERTY);

    if (this.timeOut != null) {
      return this.timeOut;
    } else if (property != null) {
      return Integer.valueOf(property);
    } else {
      return DEFAULT_TIMEOUT;
    }
  }

  private void validateIgnore(String ignore) {
    if (ignore == null) {
      throw new MunitError("Ignore expression error. The expression cannot be null");
    }

    if (ignore.isEmpty()) {
      throw new MunitError("Ignore expression error. The expression cannot be empty");
    }

    if (booleanPattern.matcher(ignore).matches()) {
      return;
    }

    MunitExpressionWrapper expressionWrapper = getExpressionWrapper();

    if (!expressionWrapper.isExpressionValid(ignore)) {
      throw new MunitError(format("Ignore expression error. The expression \"%s\" is not valid", ignore));
    }

    if (!(expressionWrapper.evaluate(ignore).getValue() instanceof Boolean)) {
      throw new MunitError(format("Ignore expression error. The expression \"%s\" should return a valid boolean", ignore));
    }
  }

}
