/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model;


import org.mule.munit.common.protocol.message.TestStatus;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Is the representation of the Suite result.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SuiteResult {

  private String name;
  private long elapsedTime;

  private List<TestResult> results = new ArrayList<>();

  public SuiteResult(String name) {
    this.name = name;
  }

  /**
   * Notifies if the Suite failed or not
   *
   * @return false if one of the tests in the suite didn't succeed.
   */
  public boolean hasSucceeded() {
    return results.stream().allMatch(r -> r.getStatus() == TestStatus.SUCCESS);
  }

  /**
   * Returns the number of failing tests in the suite
   * 
   * @return The number of test with failures in the suite
   */
  public int getNumberOfFailures() {
    return (int) results.stream().filter(r -> r.getStatus() == TestStatus.FAILURE).count();
  }

  /**
   * Returns the number of errors in the suite
   * 
   * @return The number of test with errors in the suite
   */
  public int getNumberOfErrors() {
    return getErrorTests().size();
  }

  public int getNumberOfSkipped() {
    return (int) results.stream().filter(r -> r.getStatus() == TestStatus.IGNORED).count();
  }

  public int getNumberOfTests() {
    return this.results.size();
  }

  public long getElapsedTime() {
    return elapsedTime;
  }

  public void setElapsedTime(long elapsedTime) {
    this.elapsedTime = elapsedTime;
  }

  /**
   * Returns the Suite Name
   * 
   * @return The suite name.
   */
  public String getTestName() {
    return name;
  }

  /**
   * Returns all failing tests
   * 
   * @return All the failing tests of the Suite
   */
  public List<TestResult> getFailingTests() {
    return results.stream().filter(r -> r.getStatus() == TestStatus.FAILURE).collect(Collectors.toList());
  }

  /**
   * Returns all tests in error
   * 
   * @return All the test with errors of the Suite
   */
  public List<TestResult> getErrorTests() {
    return results.stream().filter(r -> r.getStatus() == TestStatus.ERROR).collect(Collectors.toList());
  }

  public void add(TestResult result) {
    results.add(result);
  }

}
