/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;

import static com.google.common.base.Preconditions.checkNotNull;

import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.runner.model.Suite;
import org.mule.munit.runner.model.SuiteResult;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Runs a MUnit Suite
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SuiteRunner {

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private Suite suite;
  private SuiteRunEventListener suiteRunEventListener;

  public SuiteRunner(Suite suite, SuiteRunEventListener suiteRunEventListener) {
    checkNotNull(suite, "The suite must not be null");
    checkNotNull(suiteRunEventListener, "The suiteRunEvent listener must not be null");

    this.suite = suite;
    this.suiteRunEventListener = suiteRunEventListener;
  }

  /**
   * Runs the suite based on the constructor arguments
   */
  public void run() {
    long suiteStartTime = System.currentTimeMillis();
    suiteRunEventListener.notifySuiteStart(suite.getPath(), suite.getParameterization(), suite.getNumberOfTests());
    if (shouldRunSuite()) {
      doRun();
    } else {
      logger.info("MUnit suite: " + suite.getPath() + " will not be run. There are no tests to be run or all tests are ignored");
    }
    suiteRunEventListener.notifySuiteEnd(suite.getPath(), suite.getParameterization(),
                                         System.currentTimeMillis() - suiteStartTime);
  }

  private boolean shouldRunSuite() {
    return suite.getNumberOfTests() > 0 && !suite.allTestsIgnored();
  }

  private SuiteResult doRun() {
    logger.debug("About to run MUnit suite: " + suite.getPath() + " ...");
    try {
      SuiteResult suiteResult = suite.run();
      logger.debug("Tests in MUnit suite: " + suite.getPath() + " run");
      return suiteResult;
    } catch (Throwable e) {
      logger.error("Could not Run the suite: " + suite.getPath(), e);
      throw new RuntimeException("Could not Run the suite", e);
    }
  }

}
