/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.component.rules;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import java.io.Serializable;
import java.util.Optional;
import java.util.concurrent.locks.Lock;

import javax.inject.Inject;
import javax.inject.Named;

import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.api.store.ObjectDoesNotExistException;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.runtime.api.store.ObjectStoreSettings;
import org.mule.runtime.core.api.config.MuleProperties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Rule to manage temporary data.
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class TemporaryStorageRule implements TestRule, Initialisable {

  public static final String OBJECT_STORE_NAME = "_munit-temporary-storage";

  @Inject
  @Named(MuleProperties.OBJECT_STORE_MANAGER)
  protected ObjectStoreManager objectStoreManager;

  @Inject
  @Named(MuleProperties.OBJECT_LOCK_FACTORY)
  protected LockFactory lockFactory;

  private static final Logger logger = LoggerFactory.getLogger(TemporaryStorageRule.class);
  private ObjectStore<Serializable> temporaryObjectStore;

  @Override
  public void initialise() {
    ObjectStoreSettings objectStoreSettings = ObjectStoreSettings.builder().persistent(false).build();
    temporaryObjectStore = objectStoreManager.getOrCreateObjectStore(OBJECT_STORE_NAME, objectStoreSettings);
  }

  public void store(String key, Serializable value) {
    withLockedKey(temporaryObjectStore, key, objectStore -> {
      try {
        if (objectStore.contains(key)) {
          Serializable previous = objectStore.remove(key);
          logger.debug("Removing previous value [{}] for key [{}]", previous, key);
        }
        objectStore.store(key, value);
        logger.debug("Storing value [{}] for key [{}]", value, key);
        return null;
      } catch (ObjectStoreException e) {
        throw new TestRuleException(format("Unable to store value for key [%s]", key), e);
      }
    });
  }

  public Optional<Serializable> retrieve(String key) {
    return withLockedKey(temporaryObjectStore, key, store -> {
      try {
        Serializable value = store.retrieve(key);
        logger.debug("Getting value [{}] for key [{}]", value, key);
        return of(value);
      } catch (ObjectDoesNotExistException e) {
        logger.debug("No value associated with key [{}]", key);
        return empty();
      } catch (ObjectStoreException e) {
        throw new TestRuleException(format("Unable to retrieve value for key [%s]", key), e);
      }
    });
  }

  public Optional<Serializable> remove(String key) {
    return withLockedKey(temporaryObjectStore, key, store -> {
      try {
        Serializable value = store.remove(key);
        logger.debug("Removing value [{}] for key [{}]", value, key);
        return of(value);
      } catch (ObjectDoesNotExistException e) {
        logger.debug("No value associated with key [{}]", key);
        return empty();
      } catch (ObjectStoreException e) {
        throw new TestRuleException(format("Unable to remove value for key [%s]", key), e);
      }
    });
  }

  public void clear() {
    try {
      temporaryObjectStore.clear();
      logger.trace("Clearing all stored data");
    } catch (ObjectStoreException e) {
      throw new TestRuleException("Unable to clear stored data", e);
    }
  }

  @Override
  public void reset() {
    clear();
  }

  private <T> T withLockedKey(ObjectStore objectStore, String key, TemporaryStorageTask<T> task) {
    Lock lock = getKeyLock(key);
    lock.lock();
    try {
      return task.run(objectStore);
    } finally {
      lock.unlock();
    }
  }

  private Lock getKeyLock(String key) {
    return lockFactory.createLock(OBJECT_STORE_NAME + "_" + key);
  }

  @FunctionalInterface
  private interface TemporaryStorageTask<T> {

    T run(ObjectStore temporaryObjectStore);
  }

}
