/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.processors;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.util.MunitExpressionWrapper;
import org.mule.runtime.api.component.AbstractComponent;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.core.api.config.i18n.CoreMessages;
import org.mule.runtime.core.api.el.ExtendedExpressionManager;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.privileged.exception.EventProcessingException;

/**
 * Generic Munit Processor.
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class MunitProcessor extends AbstractComponent implements Initialisable, Processor {

  @Inject
  private ExtendedExpressionManager extendedExpressionManager;

  protected MunitExpressionWrapper expressionWrapper;

  public void initialise() throws InitialisationException {
    expressionWrapper = new MunitExpressionWrapper(extendedExpressionManager);
  }

  /**
   * Executes the message processor code. In case of an assertion error it throws a new exception with the mule stack trace
   * 
   * @param event The mule event to be processed.
   * @return The result mule event
   * @throws MuleException In case of error. If the assertion fails, it throws an {@link AssertionError}
   * 
   */
  public CoreEvent process(CoreEvent event) throws MuleException {
    try {
      return doProcess(event);
    } catch (AssertionError error) {
      AssertionError exception = new AssertionError(getMessage(error));
      throw exception;
    } catch (Exception e) {
      throw new EventProcessingException(CoreMessages.failedToInvoke(getProcessor()), event, e);
    }
  }

  /**
   * The method that do the actual process
   *
   * @param event * The mule Event
   * @return the processed event
   */
  protected abstract CoreEvent doProcess(CoreEvent event);

  /**
   * @return The name of the processor
   */
  protected abstract String getProcessor();

  public void setExtendedExpressionManager(ExtendedExpressionManager extendedExpressionManager) {
    this.extendedExpressionManager = extendedExpressionManager;
  }

  private String getMessage(AssertionError error) {
    String message = error.getMessage();
    if (StringUtils.isEmpty(message)) {
      return this.getProcessor();
    }
    return message;
  }
}
