/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model.builders;


import static org.mule.munit.runner.config.TestComponentLocator.CONFIG_IDENTIFIER;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.runner.component.TestComponent;
import org.mule.munit.runner.config.TestComponentLocator;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.model.Suite;
import org.mule.munit.runner.model.Test;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.munit.runner.remote.api.notifiers.DummySuiteRunEventListener;
import org.mule.runtime.ast.api.ArtifactAst;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

/**
 * Creates a MUnit Suite and its Tests
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SuiteBuilder {

  protected String suitePath;
  private final String parameterization;
  private final TestComponentLocator testComponentLocator;
  private final Optional<ArtifactAst> testArtifactAst;

  protected Set<String> tags;
  protected List<String> testNames;
  protected SuiteRunEventListener suiteRunEventListener;

  public SuiteBuilder(String suitePath, String parameterization, TestComponentLocator testComponentLocator,
                      Optional<ArtifactAst> testArtifactAst) {
    checkArgument(StringUtils.isNotBlank(suitePath), "The suitePath must not be null nor empty");
    checkNotNull(testComponentLocator, "The test component locator must not be null");

    this.suitePath = suitePath;
    this.parameterization = parameterization;
    this.suiteRunEventListener = new DummySuiteRunEventListener();
    this.testComponentLocator = testComponentLocator;
    this.testArtifactAst = testArtifactAst;
  }

  public SuiteBuilder withTestNames(List<String> testNames) {
    this.testNames = testNames;
    return this;
  }

  public SuiteBuilder withTags(Set<String> tags) {
    this.tags = tags;
    return this;
  }

  public SuiteBuilder withSuiteRunnerEventListener(SuiteRunEventListener suiteRunEventListener) {
    checkNotNull(suiteRunEventListener, "The suiteRunEventListener must not be null");
    this.suiteRunEventListener = suiteRunEventListener;
    return this;
  }

  /**
   * Builds the Suite with a particular suite name, based on the mule context
   *
   * @return The Suite Object
   */
  public Suite build() {
    testComponentLocator.initializeComponents(suitePath);
    MunitModule munitModule = testComponentLocator.lookupMunitModule()
        .orElseThrow(() -> new IllegalStateException("Missing element [" + CONFIG_IDENTIFIER + "] in file " + suitePath));

    Suite suite = new Suite(suitePath, parameterization);
    suite.setSuiteRunEventListener(suiteRunEventListener);

    suite.setBeforeSuite(testComponentLocator.lookupBeforeSuite().orElse(null));
    createTests(munitModule).forEach(suite::addTest);
    suite.setAfterSuite(testComponentLocator.lookupAfterSuite().orElse(null));

    return suite;
  }

  private List<Test> createTests(MunitModule munitModule) {
    BeforeTest before = testComponentLocator.lookupBeforeTest().orElse(null);
    Collection<TestComponent> testComponents = testComponentLocator.lookupTests();
    AfterTest after = testComponentLocator.lookupAfterTest().orElse(null);

    TestRunFilter filter = new TestRunFilter();

    return testComponents.stream()
        .filter(testComponent -> filter.shouldRunTest(testComponent, testNames, tags))
        .map(testComponent -> test(before, testComponent, after, munitModule)).collect(Collectors.toList());
  }

  /**
   * Create a test
   *
   * @param beforeTest MUnit flow to be run before the test
   * @param test MUnit Flow that represents the test
   * @param afterTest MUnit flow to be run after the test
   * @return The Test Object
   */
  protected Test test(BeforeTest beforeTest, TestComponent test, AfterTest afterTest, MunitModule munitModule) {
    return new Test(beforeTest, test, afterTest, munitModule);
  }

}
