/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote.api.notifiers;

import static java.lang.String.format;
import static java.util.stream.Collectors.joining;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.repeat;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.message.TestStatus;

/**
 * <p>
 * Prints friendly messages of the test results through a given {@link PrintStream}
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class StreamNotifier implements RunEventListener {

  private static final String SUITE_TOKEN = "=";
  private static final String TEST_TOKEN = "+";
  private static final Integer LINE_LENGTH = 80;

  private PrintStream out;
  private int tests = 0;
  private int failures = 0;
  private int errors = 0;
  private int skipped = 0;

  public StreamNotifier(PrintStream out) {
    this.out = out;
  }

  @Override
  public void notifyTestStart(String name, String description, boolean isIgnored) {
    String descriptionMessage = isBlank(description) ? EMPTY : " - " + description;
    String title = repeat(TEST_TOKEN, LINE_LENGTH);
    String message;
    if (isIgnored) {
      message = "Ignoring test: " + name + descriptionMessage;
    } else {
      message = "Running test: " + name + descriptionMessage;
    }
    out.println(title);
    out.println(generateMessage(TEST_TOKEN + " ", TEST_TOKEN, message));
    out.println(title);
    out.flush();
  }

  @Override
  public void notifySuiteStart(String suite, String parameterization, int numberOfTests) {
    String suiteMessage = "Running suite: " + suite;
    if (isNotBlank(parameterization)) {
      suiteMessage += "[" + parameterization + "]";
    }
    String title = repeat(SUITE_TOKEN, LINE_LENGTH);
    out.println(title);
    out.println(generateMessage(repeat(SUITE_TOKEN, 3) + " ", repeat(SUITE_TOKEN, 3), suiteMessage));
    out.println(title);
    out.flush();
  }

  @Override
  public void notifyTestEnd(String name, String stackTrace, TestStatus status, long elapsedTime) {
    tests++;
    switch (status) {
      case SUCCESS:
        break;
      case ERROR:
        errors++;
        break;
      case FAILURE:
        failures++;
        break;
      case IGNORED:
        skipped++;
        break;
    }
    if (StringUtils.isNotBlank(stackTrace)) {
      out.println(repeat(TEST_TOKEN, LINE_LENGTH));
      out.println(stackTrace);
      out.println(repeat(TEST_TOKEN, LINE_LENGTH));
    }
    out.flush();
  }

  @Override
  public void notifySuiteEnd(String suite, String parameterization, long elapsedTime) {
    String title = "Tests run: %d - Failed: %d - Errors: %d - Skipped: %d - Time elapsed: %.2f sec";
    String titleFrame = StringUtils.repeat(SUITE_TOKEN, LINE_LENGTH);
    out.println(titleFrame);
    out.println(generateMessage(SUITE_TOKEN + " ", SUITE_TOKEN,
                                format(title, tests, failures, errors, skipped, (elapsedTime / 1000.0))));
    out.println(titleFrame);
    out.flush();
  }

  private String generateMessage(String prefix, String suffix, String message) {
    return splitInLines(message, LINE_LENGTH - (prefix.length() + suffix.length())).stream()
        .map(line -> prefix + line + repeat(" ", LINE_LENGTH - (line.length() + prefix.length() + suffix.length())) + suffix)
        .collect(joining(System.lineSeparator()));
  }

  private List<String> splitInLines(String line, int length) {
    List<String> list = new ArrayList<String>((line.length() + length - 1) / length);

    for (int start = 0; start < line.length(); start += length) {
      list.add(line.substring(start, Math.min(line.length(), start + length)));
    }

    return list;
  }

}
