/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.model.builders;

import static org.mule.munit.common.api.model.stereotype.MUnitStereotypes.TEST_PROCESSOR;
import static org.mule.munit.common.api.util.VersionUtils.isAtLeastMinMuleVersion;
import static org.mule.munit.common.util.Preconditions.checkArgument;
import static org.mule.munit.common.util.Preconditions.checkNotNull;
import static org.mule.munit.runner.config.TestComponentLocator.CONFIG_IDENTIFIER;
import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.deployment.meta.Product.getProductByName;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;

import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.Stream.concat;

import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.runner.component.TestComponent;
import org.mule.munit.runner.config.TestComponentLocator;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.model.Suite;
import org.mule.munit.runner.model.Test;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.munit.runner.remote.api.notifiers.DummySuiteRunEventListener;
import org.mule.runtime.api.deployment.meta.Product;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.stereotype.HasStereotypeModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.core.api.config.MuleManifest;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

/**
 * Creates a MUnit Suite and its Tests
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class SuiteBuilder {

  private static BiFunction<ComponentAst, String, ComponentParameterAst> AST_PARAM_FETCHER;

  static {
    try {
      // ast 1.x
      Method getParameterMethod = ComponentAst.class.getDeclaredMethod("getParameter", String.class, String.class);

      AST_PARAM_FETCHER = (comp, paramName) -> {
        try {
          return ((ComponentParameterAst) (getParameterMethod.invoke(comp, DEFAULT_GROUP_NAME, paramName)));
        } catch (InvocationTargetException e) {
          throw new MuleRuntimeException(e.getCause());
        } catch (IllegalAccessException e) {
          throw new MuleRuntimeException(e);
        }
      };
    } catch (NoSuchMethodException nsme) {
      try {
        // ast 0.8
        Method getParameterMethod = ComponentAst.class.getDeclaredMethod("getParameter", String.class);

        AST_PARAM_FETCHER = (comp, paramName) -> {
          try {
            return ((ComponentParameterAst) (getParameterMethod.invoke(comp, paramName)));
          } catch (InvocationTargetException e) {
            throw new MuleRuntimeException(e.getCause());
          } catch (IllegalAccessException e) {
            throw new MuleRuntimeException(e);
          }
        };
      } catch (NoSuchMethodException e) {
        throw new MuleRuntimeException(createStaticMessage("No compatible ArtifactAST library available"), e);
      }
    }

  }

  protected String suitePath;
  private final String parameterization;
  private final TestComponentLocator testComponentLocator;
  private final Optional<ArtifactAst> testArtifactAst;

  protected Set<String> tags;
  protected List<RunConfiguration.Test> testNames;
  protected SuiteRunEventListener suiteRunEventListener;

  public SuiteBuilder(String suitePath, String parameterization, TestComponentLocator testComponentLocator,
                      Optional<ArtifactAst> testArtifactAst) {
    checkArgument(StringUtils.isNotBlank(suitePath), "The suitePath must not be null nor empty");
    checkNotNull(testComponentLocator, "The test component locator must not be null");

    this.suitePath = suitePath;
    this.parameterization = parameterization;
    this.suiteRunEventListener = new DummySuiteRunEventListener();
    this.testComponentLocator = testComponentLocator;
    this.testArtifactAst = testArtifactAst;
  }

  public SuiteBuilder withTestNames(List<RunConfiguration.Test> testNames) {
    this.testNames = testNames;
    return this;
  }

  public SuiteBuilder withTags(Set<String> tags) {
    this.tags = tags;
    return this;
  }

  public SuiteBuilder withSuiteRunnerEventListener(SuiteRunEventListener suiteRunEventListener) {
    checkNotNull(suiteRunEventListener, "The suiteRunEventListener must not be null");
    this.suiteRunEventListener = suiteRunEventListener;
    return this;
  }

  /**
   * Builds the Suite with a particular suite name, based on the mule context
   *
   * @return The Suite Object
   */
  public Suite build() {
    Suite suite = new Suite(suitePath, parameterization);
    suite.setSuiteRunEventListener(suiteRunEventListener);

    if (testArtifactAst.isPresent()) {
      ComponentAst munitModuleComponent = testArtifactAst.get().topLevelComponentsStream()
          .filter(c -> c.getLocation().getFileName().map(suitePath::equals).orElse(false))
          .filter(c -> c.getIdentifier().equals(buildFromStringRepresentation(CONFIG_IDENTIFIER)))
          .findAny()
          .orElseThrow(() -> new IllegalStateException("Missing element [" + CONFIG_IDENTIFIER + "] in file " + suitePath));


      Optional<String> minMuleVersion = ofNullable((String) AST_PARAM_FETCHER.apply(munitModuleComponent, "minMuleVersion")
          .getValue().getRight());
      String requiredProductName = (String) AST_PARAM_FETCHER.apply(munitModuleComponent, "requiredProduct")
          .getValue().getRight();

      Product actualProduct = getProductByName(MuleManifest.getProductName());
      String actualVersion = MuleManifest.getProductVersion();

      Product requiredProduct = requiredProductName != null ? Product.valueOf(requiredProductName) : Product.MULE;

      if (actualProduct.supports(requiredProduct)
          && minMuleVersion.map(mmv -> isAtLeastMinMuleVersion(actualVersion, mmv))
              .orElse(true)) {
        TestRunFilter filter = new TestRunFilter();

        // Need the stereotype information from the components in the ast to infer any implicit tags
        if (testArtifactAst.get().topLevelComponentsStream()
            .filter(c -> c.getLocation().getFileName().map(suitePath::equals).orElse(false))
            .anyMatch(c -> filter.shouldRunTest(c.getComponentId().get(), getTestTags(c), testNames, tags, suitePath))) {
          testComponentLocator.initializeComponents(suitePath);
          MunitModule munitModule = testComponentLocator.lookupMunitModule()
              .orElseThrow(() -> new IllegalStateException("Missing element [" + CONFIG_IDENTIFIER + "] in file " + suitePath));

          suite.setBeforeSuite(testComponentLocator.lookupBeforeSuite().orElse(null));
          createTests(munitModule, true, false).forEach(suite::addTest);
          suite.setAfterSuite(testComponentLocator.lookupAfterSuite().orElse(null));
        }
      }
    } else {
      testComponentLocator.initializeComponents(suitePath);
      MunitModule munitModule = testComponentLocator.lookupMunitModule()
          .orElseThrow(() -> new IllegalStateException("Missing element [" + CONFIG_IDENTIFIER + "] in file " + suitePath));

      suite.setBeforeSuite(testComponentLocator.lookupBeforeSuite().orElse(null));
      createTests(munitModule, true, true).forEach(suite::addTest);
      suite.setAfterSuite(testComponentLocator.lookupAfterSuite().orElse(null));
    }

    return suite;
  }

  private Set<String> getTestTags(ComponentAst testComponent) {
    Stream<String> implicitTags = testComponent.recursiveStream()
        .map(c -> c.getModel(HasStereotypeModel.class)
            .flatMap(hsm -> ofNullable(hsm.getStereotype())))
        .flatMap(s -> s.map(Stream::of).orElse(Stream.empty()))
        .filter(stereotype -> stereotype.isAssignableTo(TEST_PROCESSOR))
        .map(stereotype -> stereotype.getNamespace() + ":" + stereotype.getType());

    ComponentParameterAst tagsParam = AST_PARAM_FETCHER.apply(testComponent, "tags");
    if (tagsParam == null) {
      return implicitTags.collect(toSet());
    }
    Object tagsParamValue = tagsParam.getValue().getRight();
    if (tagsParamValue == null) {
      return implicitTags.collect(toSet());
    }

    Stream<String> explicitTags = Stream.of(((String) tagsParamValue).split(","));
    return concat(implicitTags, explicitTags).collect(toSet());
  }

  private List<Test> createTests(MunitModule munitModule, boolean doFilter, boolean filterByTag) {
    BeforeTest before = testComponentLocator.lookupBeforeTest().orElse(null);
    Collection<TestComponent> testComponents = testComponentLocator.lookupTests();
    AfterTest after = testComponentLocator.lookupAfterTest().orElse(null);

    TestRunFilter filter = new TestRunFilter();
    if (filterByTag) {
      return testComponents.stream()
          .filter(testComponent -> !doFilter
              || filter.shouldRunTest(testComponent.getName(), testComponent.getTags(), testNames, tags, suitePath))
          .map(testComponent -> test(before, testComponent, after, munitModule))
          .collect(toList());
    } else {
      return testComponents.stream()
          .filter(testComponent -> !doFilter
              || filter.shouldRunTest(testComponent.getName(), testNames, suitePath))
          .map(testComponent -> test(before, testComponent, after, munitModule))
          .collect(toList());
    }
  }

  /**
   * Create a test
   *
   * @param beforeTest MUnit flow to be run before the test
   * @param test       MUnit Flow that represents the test
   * @param afterTest  MUnit flow to be run after the test
   * @return The Test Object
   */
  protected Test test(BeforeTest beforeTest, TestComponent test, AfterTest afterTest, MunitModule munitModule) {
    return new Test(beforeTest, test, afterTest, munitModule);
  }

}
