/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.config;

import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.AFTER_SUITE;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.AFTER_TEST;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.BEFORE_SUITE;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.BEFORE_TEST;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.CONFIG;
import static org.mule.munit.runner.config.MunitComponentBuildingDefinitionProvider.TEST;
import static org.mule.munit.runner.config.MunitXmlNamespaceInfoProvider.NAME_SPACE;
import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import org.mule.munit.runner.component.TestComponent;
import org.mule.munit.runner.component.TestComponentInfoProvider;
import org.mule.munit.runner.flow.AfterSuite;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeSuite;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.config.api.LazyComponentInitializer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Initializes and looks up test components such as tests, before/after test/suites and the config.
 * 
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class TestComponentLocator {

  public static final String CONFIG_IDENTIFIER = NAME_SPACE + ":" + CONFIG;
  public static final String BEFORE_SUITE_IDENTIFIER = NAME_SPACE + ":" + BEFORE_SUITE;
  public static final String AFTER_SUITE_IDENTIFIER = NAME_SPACE + ":" + AFTER_SUITE;
  public static final String BEFORE_TEST_IDENTIFIER = NAME_SPACE + ":" + BEFORE_TEST;
  public static final String AFTER_TEST_IDENTIFIER = NAME_SPACE + ":" + AFTER_TEST;
  public static final String TEST_IDENTIFIER = NAME_SPACE + ":" + TEST;

  private final ConfigurationComponentLocator configurationComponentLocator;
  private final LazyComponentInitializer lazyComponentInitializer;
  private final List<ComponentIdentifier> testComponentIdentifiers = new ArrayList<>();
  private final Set<String> testNamespaces = new HashSet<>();

  public TestComponentLocator(ConfigurationComponentLocator configurationComponentLocator,
                              LazyComponentInitializer lazyComponentInitializer,
                              Collection<TestComponentInfoProvider> testComponentInfoProviders) {
    this.configurationComponentLocator = configurationComponentLocator;
    this.lazyComponentInitializer = lazyComponentInitializer;
    testComponentInfoProviders.forEach(testComponentInfoProvider -> {
      ComponentIdentifier testComponentIdentifier = testComponentInfoProvider.getComponentIdentifier();
      testComponentIdentifiers.add(testComponentIdentifier);
      testNamespaces.add(testComponentIdentifier.getNamespace());
    });
  }

  public void initializeComponents(String suitePath) {
    lazyComponentInitializer.initializeComponents(componentLocation -> belongsToSuite(suitePath, componentLocation)
        && isTestNamespace(componentLocation));
  }

  private boolean isTestNamespace(ComponentLocation componentLocation) {
    return testNamespaces.contains(componentLocation.getComponentIdentifier().getIdentifier().getNamespace());
  }

  private Boolean belongsToSuite(String suitePath, ComponentLocation componentLocation) {
    return componentLocation.getFileName().map(suitePath::equals).orElse(false);
  }

  public Optional<MunitModule> lookupMunitModule() {
    final ComponentIdentifier componentIdentifier = buildFromStringRepresentation(CONFIG_IDENTIFIER);
    List<Component> components = configurationComponentLocator.find(componentIdentifier);
    if (!components.isEmpty()) {
      return of((MunitModule) components.get(0));
    }
    return empty();
  }

  public Optional<BeforeSuite> lookupBeforeSuite() {
    List<Component> components = configurationComponentLocator.find(buildFromStringRepresentation(BEFORE_SUITE_IDENTIFIER));
    if (!components.isEmpty()) {
      return of((BeforeSuite) components.get(0));
    }
    return empty();
  }

  public Optional<AfterSuite> lookupAfterSuite() {
    List<Component> components = configurationComponentLocator.find(buildFromStringRepresentation(AFTER_SUITE_IDENTIFIER));
    if (!components.isEmpty()) {
      return of((AfterSuite) components.get(0));
    }
    return empty();
  }

  public Optional<BeforeTest> lookupBeforeTest() {
    List<Component> components = configurationComponentLocator.find(buildFromStringRepresentation(BEFORE_TEST_IDENTIFIER));
    if (!components.isEmpty()) {
      return of((BeforeTest) components.get(0));
    }
    return empty();
  }

  public Optional<AfterTest> lookupAfterTest() {
    List<Component> components = configurationComponentLocator.find(buildFromStringRepresentation(AFTER_TEST_IDENTIFIER));
    if (!components.isEmpty()) {
      return of((AfterTest) components.get(0));
    }
    return empty();
  }

  public Collection<TestComponent> lookupTests() {
    return testComponentIdentifiers.stream().map(testIdentifier -> configurationComponentLocator.find(testIdentifier))
        .flatMap(List::stream).filter(testComponent -> testComponent instanceof TestComponent)
        .map(testComponent -> (TestComponent) testComponent).collect(toList());
  }

}
