/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote.api.server;

import static org.mule.munit.common.protocol.message.MessageField.*;
import static org.mule.munit.common.protocol.message.MessageID.RUN_SUITE;

import static java.util.Optional.ofNullable;

import static org.mule.munit.common.util.Collections.newHashSet;

import com.google.gson.reflect.TypeToken;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.common.protocol.listeners.SuiteRunEventListener;
import org.mule.munit.common.protocol.message.RunMessage;
import org.mule.munit.common.util.StackTraceUtil;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.runner.SuiteRunner;
import org.mule.munit.runner.config.TestComponentLocator;
import org.mule.munit.runner.model.Suite;
import org.mule.munit.runner.model.builders.SuiteBuilder;
import org.mule.munit.runner.remote.api.notifiers.ObjectOutputNotifier;
import org.mule.munit.runner.remote.api.notifiers.StreamNotifier;
import org.mule.runtime.ast.api.ArtifactAst;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.Type;
import java.util.*;

import com.google.gson.Gson;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * The goal of this class is to parse incoming message from a socket and run the suite according to the message parameters
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunMessageHandler {

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private final ObjectInput in;
  private final ObjectOutput out;
  private final TestComponentLocator testComponentLocator;
  private final Optional<ArtifactAst> testArtifactAst;

  public RunMessageHandler(ObjectInput in, ObjectOutput out, TestComponentLocator testComponentLocator,
                           Optional<ArtifactAst> testArtifactAst) {
    this.testComponentLocator = testComponentLocator;
    this.testArtifactAst = testArtifactAst;
    this.in = in;
    this.out = out;
  }

  public void handle() throws IOException, ClassNotFoundException {
    String message = (String) in.readObject();
    RunMessage runMessage = new Gson().fromJson(message, RunMessage.class);
    SuiteRunEventListener listener = buildListenerContainer(runMessage.get(RUN_TOKEN_KEY));

    if (RUN_SUITE == runMessage.getId()) {
      parseAndRun(runMessage, listener);
    }
  }

  protected void parseAndRun(RunMessage runMessage, SuiteRunEventListener listener) {
    try {
      Suite suite = parseSuiteMessage(runMessage, listener);
      runSuite(suite, listener);
    } catch (Throwable e) {
      e.printStackTrace();
      listener.notifySuiteUnexpectedError(getSuitePath(runMessage), StackTraceUtil.getStackTrace(e));
    }
  }

  private Suite parseSuiteMessage(RunMessage runMessage, SuiteRunEventListener listener) {
    String suitePath = getSuitePath(runMessage);
    String parameterization = getParameterization(runMessage);
    List<RunConfiguration.Test> testNames = getTestNames(runMessage);
    Set<String> tags = getTags(runMessage);
    return buildSuite(suitePath, parameterization, testNames, tags, listener);
  }

  protected void runSuite(Suite suite, SuiteRunEventListener listener) {
    SuiteRunner runner = new SuiteRunner(suite, listener);
    runner.run();
  }

  protected Suite buildSuite(String suitePath, String parameterization, List<RunConfiguration.Test> testNames,
                             Set<String> tags,
                             SuiteRunEventListener listener) {
    SuiteBuilder suiteBuilder = new SuiteBuilder(suitePath, parameterization, testComponentLocator, testArtifactAst)
        .withTestNames(testNames)
        .withTags(tags)
        .withSuiteRunnerEventListener(listener);
    return suiteBuilder.build();
  }

  private String getSuitePath(RunMessage runMessage) {
    return runMessage.get(MUNIT_SUITE_KEY);
  }

  private String getParameterization(RunMessage runMessage) {
    return ofNullable(runMessage.get(PARAMETERIZATION_KEY)).orElse(StringUtils.EMPTY);
  }

  private SuiteRunEventListener buildListenerContainer(String runToken) {
    RunEventListenerContainer listenerDecorator = new RunEventListenerContainer();
    listenerDecorator.addNotificationListener(new ObjectOutputNotifier(runToken, out));
    listenerDecorator.addNotificationListener(new StreamNotifier(System.out));
    return listenerDecorator;
  }

  private List<RunConfiguration.Test> getTestNames(RunMessage runMessage) {
    String testNames =
        Optional.ofNullable(runMessage.get(TEST_NAMES_WITH_SUITE_KEY)).orElseGet(() -> runMessage.get(TEST_NAMES_KEY));
    Type typeOfHashMap = new TypeToken<List<RunConfiguration.Test>>() {}.getType();
    return new Gson().fromJson(testNames, typeOfHashMap);
  }

  private Set<String> getTags(RunMessage runMessage) {
    String tags = runMessage.get(TAGS_KEY);
    if (StringUtils.isBlank(tags)) {
      return Collections.emptySet();
    }
    return newHashSet(tags.split(SEPARATOR_TOKEN));
  }
}
