/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.component.properties;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import java.util.Objects;
import java.util.Optional;

import org.mule.munit.common.util.FreePortFinder;
import org.mule.runtime.config.api.dsl.model.properties.ConfigurationPropertiesProvider;
import org.mule.runtime.config.api.dsl.model.properties.ConfigurationProperty;

/**
 * {@link ConfigurationPropertiesProvider} that provides a {@link ConfigurationProperty} binding a free port to the given property
 * name
 * 
 * @since 2.2.0
 * @author Mulesoft Inc.
 */
public class DynamicPortPropertiesProvider implements ConfigurationPropertiesProvider {

  private static final Integer DEFAULT_MIN_PORT = 1;
  private static final Integer DEFAULT_MAX_PORT = 65535;

  private String propertyName;
  private DynamicPortConfigurationProperty configurationProperty;

  public DynamicPortPropertiesProvider(String propertyName, Integer min, Integer max) {
    this.propertyName = propertyName;
    this.configurationProperty = new DynamicPortConfigurationProperty(this, propertyName, findPort(min, max));
  }

  @Override
  public Optional<ConfigurationProperty> getConfigurationProperty(String configurationAttributeKey) {
    return Objects.equals(propertyName, configurationAttributeKey) ? of(configurationProperty) : empty();
  }

  private static class DynamicPortConfigurationProperty implements ConfigurationProperty {

    private Object source;
    private String propertyName;
    private Integer port;

    DynamicPortConfigurationProperty(Object source, String propertyName, Integer port) {
      this.source = source;
      this.propertyName = propertyName;
      this.port = port;
    }

    @Override
    public Object getSource() {
      return source;
    }

    @Override
    public Object getRawValue() {
      return String.valueOf(port);
    }

    @Override
    public String getKey() {
      return propertyName;
    }
  }

  @Override
  public String getDescription() {
    return format("<munit:dynamic-port propertyName=\"%s\" >", propertyName);
  }


  private Integer findPort(Integer min, Integer max) {
    Integer minPort = ofNullable(min).orElse(DEFAULT_MIN_PORT);
    Integer maxPort = ofNullable(max).orElse(DEFAULT_MAX_PORT);
    if (minPort > maxPort) {
      throw new IllegalArgumentException(format("Minimum port %d cannot be bigger than maximum port %d", minPort, maxPort));
    }
    return new FreePortFinder(minPort, maxPort).find();
  }
}
