/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.assertion.functions;

import java.nio.charset.Charset;

import org.mule.munit.assertion.api.matchers.Matcher;
import org.mule.munit.assertion.api.matchers.core.BothMatcher;
import org.mule.munit.assertion.api.matchers.core.EitherMatcher;
import org.mule.munit.assertion.api.matchers.core.EqualToMatcher;
import org.mule.munit.assertion.api.matchers.core.NotMatcher;
import org.mule.munit.assertion.api.matchers.core.NotNullValueMatcher;
import org.mule.munit.assertion.api.matchers.core.NullValueMatcher;
import org.mule.munit.assertion.api.matchers.core.WithEncodingMatcher;
import org.mule.munit.assertion.api.matchers.core.WithMediaTypeMatcher;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

/**
 * Core matchers to perform assertions
 * 
 * @since 2.0.0
 * @author Mulesoft Inc.
 */
public class CoreMatcherFunctions {


  @Summary("Matches if expression equals a value")
  public Matcher equalTo(Object value) {
    return new EqualToMatcher(value);
  }

  @Summary("Matches if expression value is not null")
  public Matcher notNullValue() {
    return new NotNullValueMatcher();
  }

  @Summary("Matches if expression value is null")
  public Matcher nullValue() {
    return new NullValueMatcher();
  }

  @Summary("Matches if expression does not match the given matcher")
  public Matcher not(Matcher matcher) {
    return new NotMatcher(matcher);
  }

  @Summary("Matches if expression matches both given matchers")
  public Matcher both(Matcher first, Matcher second) {
    return new BothMatcher(first, second);
  }

  @Summary("Matches if expression matches at least one of the given matchers")
  public Matcher either(Matcher first, Matcher second) {
    return new EitherMatcher(first, second);
  }

  @Summary("Matches if the expression has a particular encoding")
  public Matcher withEncoding(@Optional @Example("UTF-8") String encoding) {
    return new WithEncodingMatcher(encoding == null ? null : Charset.forName(encoding));
  }

  @Summary("Matches if the expression has a particular media type")
  public Matcher withMediaType(@Example("text/xml") String mediaType) {
    return new WithMediaTypeMatcher(mediaType == null ? null : MediaType.parse(mediaType));
  }

}
