/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.mock;

import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

import java.util.List;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.model.Attribute;
import org.mule.munit.common.model.Event;
import org.mule.munit.mock.MockModule;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.RouterCompletionCallback;

/**
 * <p>
 * Operations to perform mocking
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MockOperations {

  @Inject
  private Registry registry;

  private MockModule mockModule = new MockModule();

  @Summary("Mock the Processor when it matches processor name and attributes")
  public void mockWhen(@Example("http:request") String processor,
                       @Optional @Expression(NOT_SUPPORTED) List<Attribute> withAttributes,
                       @Optional @Expression(NOT_SUPPORTED) Event thenReturn) {
    validateErrorInEvent(thenReturn);

    mockModule.setRegistry(registry);
    mockModule.when(processor, withAttributes, thenReturn);
  }

  @Summary("Verify that a processor is called")
  public void verifyCall(@Example("mule:logger") String processor,
                         @Optional @Expression(NOT_SUPPORTED) List<Attribute> withAttributes,
                         @Optional Integer times,
                         @Optional Integer atLeast,
                         @Optional Integer atMost) {

    if (null == times && null == atLeast && null == atMost) {
      times = 1;
    }
    mockModule.setRegistry(registry);
    mockModule.verifyCall(processor, withAttributes, times, atLeast, atMost);
  }

  @Summary("Allows to take actions over the event before and after the execution of a processor")
  public void spy(String processor,
                  @Optional @NullSafe @Expression(NOT_SUPPORTED) List<Attribute> withAttributes,
                  @Optional BeforeCall beforeCall,
                  @Optional AfterCall afterCall,
                  RouterCompletionCallback callback) {
    mockModule.setRegistry(registry);
    mockModule.spy(processor, withAttributes,
                   beforeCall == null ? null : beforeCall.getChain(),
                   afterCall == null ? null : afterCall.getChain());
    callback.success(Result.builder().build());
  }

  protected void setRegistry(Registry registry) {
    this.registry = registry;
  }

  protected void setMockModule(MockModule mockModule) {
    this.mockModule = mockModule;
  }

  private void validateErrorInEvent(Event thenReturn) {
    if (thenReturn != null && thenReturn.getError() != null) {
      if (StringUtils.isNotBlank(thenReturn.getError().getTypeId()) && thenReturn.getError().getCause() != null) {
        throw new IllegalArgumentException("Mocked event's error failure. The attributes typeId and cause are mutually exclusive. Please define only one of them");
      }
    }
  }
}
