/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.assertion;


import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.munit.assertion.api.TypedValue.fromMuleTypedValue;
import static org.mule.runtime.api.meta.ExpressionSupport.REQUIRED;

import org.apache.commons.lang3.StringUtils;
import org.junit.Assert;
import org.mule.munit.assertion.api.MunitAssertion;
import org.mule.munit.assertion.api.matchers.Matcher;
import org.mule.munit.assertion.internal.AssertModule;
import org.mule.munit.assertion.internal.HamcrestFactory;
import org.mule.munit.common.exception.MunitError;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.ClassValue;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

/**
 * <p>
 * Operations to perform assertions
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class AssertOperations {

  private static final String RUN_CUSTOM_INTERFACE_ERROR =
      "The run-custom operation requires a class implementing %s. Please provide one.";

  private static final String RUN_CUSTOM_ERROR = "Unable to create custom assertion";
  private static final String MUNIT_ASSERTION_CLASS = "org.mule.munit.assertion.api.MunitAssertion";

  private AssertModule assertModule = new AssertModule();

  @Summary("Perform an assertion over an expression")
  public void assertThat(@Example("#[payload]") TypedValue<?> expression,
                         @Expression(REQUIRED) @Example("#[MunitTools::notNullValue()]") Matcher is,
                         @Optional(defaultValue = StringUtils.EMPTY) String message) {
    assertModule.assertThat(message, expression, HamcrestFactory.create(is));
  }

  @Summary("Fail with an assertion")
  public void fail(@Optional(defaultValue = StringUtils.EMPTY) String message) {
    Assert.fail(message);
  }

  @Summary("Run a custom assertion")
  public void runCustom(@Example("com.example.CustomAssertion") @ClassValue(
      extendsOrImplements = MUNIT_ASSERTION_CLASS) String assertion,
                        @Example("#[payload]") TypedValue<Object> expression,
                        @Optional Object params) {
    if (isBlank(assertion)) {
      throw new MunitError(String.format(RUN_CUSTOM_INTERFACE_ERROR, MUNIT_ASSERTION_CLASS));
    }
    try {
      MunitAssertion munitAssertion = (MunitAssertion) Class.forName(assertion).newInstance();
      munitAssertion.execute(fromMuleTypedValue(expression), params);
    } catch (ClassCastException e) {
      throw new MunitError(String.format(RUN_CUSTOM_INTERFACE_ERROR, MUNIT_ASSERTION_CLASS));
    } catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
      throw new MunitError(RUN_CUSTOM_ERROR, e);
    }
  }

  protected void setAssertModule(AssertModule assertModule) {
    this.assertModule = assertModule;
  }

}
