/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.util.store;

import static org.mule.munit.tools.MunitToolsErrorDefinition.INVALID_KEY;
import org.mule.munit.runner.component.rules.TemporaryStorageRule;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;

import java.io.Serializable;

import javax.inject.Inject;

/**
 * Operations to store temporary data
 * 
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class StorageOperations {

  @Inject
  protected TemporaryStorageRule storageRule;

  /**
   * Stores the given {@code value} using the given {@code key}.
   * <p>
   * This operation can be used either for storing new values or updating existing ones. This operation is synchronized on the key
   * level. No other operation will be able to access the same key.
   *
   * @param key the key of the {@code value} to be stored
   * @param value the value to be stored
   * @param failIfPresent whether to fail or update the pre existing value if the {@code key} already exists on the store
   */
  @Summary("Stores the given data associated with the given key")
  @Throws(StoreErrorProvider.class)
  public void store(String key,
                    @Optional(defaultValue = "#[payload]") @Content TypedValue<Serializable> value,
                    @Optional(defaultValue = "false") boolean failIfPresent) {
    validateKey(key);
    if (failIfPresent) {
      storageRule.retrieve(key)
          .ifPresent(previous -> {
            throw new KeyAlreadyExistsException(key);
          });
    }
    storageRule.store(key, value);
  }

  /**
   * Retrieves the value stored for the given {@code key}.
   * <p>
   * If no value exists for the {@code key}, then a {@code MUNIT-TOOLS:MISSING_KEY} error will be thrown. This operations is
   * synchronized on the key level. No other operation will be able to access the same key while this operation is running
   *
   * @param key the key of the {@code value} to be retrieved
   * @return The stored value
   */
  @Summary("Retrieves the data associated with the given key")
  @Throws(RetrieveErrorProvider.class)
  public Result<Serializable, Void> retrieve(String key) {
    validateKey(key);
    return asResult(storageRule.retrieve(key).orElseThrow(() -> new MissingKeyException(key)));
  }

  /**
   * Removes the value associated to the given {@code key}.
   * <p>
   * If no value exist for the key, then a {@code MUNIT-TOOLS:MISSING_KEY} error will be thrown.This operation is synchronized on
   * the key level. No other operation will be able to access the same key while this operation is running.
   *
   * @param key the key of the object to be removed
   * @return The storage value
   */
  @Summary("Removes the data associated with the given key")
  @Throws(RetrieveErrorProvider.class)
  public Result<Serializable, Void> remove(String key) {
    validateKey(key);
    return asResult(storageRule.remove(key).orElseThrow(() -> new MissingKeyException(key)));
  }

  /**
   * Removes all stored data
   */
  @Summary("Clears all stored data")
  public void clearStoredData() {
    storageRule.clear();
  }

  private Result<Serializable, Void> asResult(Object value) {
    TypedValue<Serializable> typedValue = value instanceof TypedValue
        ? (TypedValue<Serializable>) value
        : new TypedValue<>((Serializable) value, DataType.fromType(value.getClass()));

    return Result.<Serializable, Void>builder()
        .output(typedValue.getValue())
        .mediaType(typedValue.getDataType().getMediaType())
        .build();
  }

  private void validateKey(String key) {
    if (key == null || key.trim().length() == 0) {
      throw new ModuleException(INVALID_KEY, new IllegalArgumentException("Key cannot be null nor empty"));
    }
  }
}
