/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.mock;

import static org.mule.munit.tools.mock.StreamingUtils.resolveCursorProviders;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;

import org.mule.munit.common.api.model.Answer;
import org.mule.munit.common.api.model.Attribute;
import org.mule.munit.common.api.model.Event;
import org.mule.munit.common.api.model.FlowName;
import org.mule.munit.common.api.model.stereotype.MUnitMockStereotype;
import org.mule.munit.mock.MockModule;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.NullSafe;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.annotation.param.stereotype.Stereotype;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.RouterCompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.util.List;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;

/**
 * <p>
 * Operations to perform mocking
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
@Stereotype(MUnitMockStereotype.class)
public class MockOperations {

  @Inject
  private Registry registry;

  private MockModule mockModule = new MockModule();

  /**
   * Mocks the {@code processor} that has the given {@code withAttributes}, replacing it with the given {@code answer}.
   * <p>
   * During the test execution, if any processor tries to be executed, and it matches the {@code processor} name and the
   * {@code withAttributes}, its execution its not performed, and instead the {@code answer} is used. If
   * {@link Answer#getThenReturn() thenReturn} is specified, then that event is returned, or if {@link Answer#getThenCall()
   * thenCall} is specified, the given flow is invoked, and the resulting event is returned. If no {@link Answer} is specified,
   * the incoming event is returned.
   *
   * @param processor identifier of the processor to mock
   * @param withAttributes additional attributes to identify the processor
   * @param answer {@link Answer} to be used when mocking the processor
   */
  @Summary("Mock the Processor when it matches processor name and attributes")
  public void mockWhen(@Example("http:request") String processor,
                       @Optional @Expression(NOT_SUPPORTED) List<Attribute> withAttributes,
                       @ParameterGroup(name = Answer.NAME) Answer answer, StreamingHelper streamingHelper) {

    Event thenReturn = answer.getThenReturn();
    FlowName thenCall = answer.getThenCall();

    mockModule.setRegistry(registry);

    if (thenCall != null) {
      mockModule.when(processor, withAttributes, thenCall);
    } else {
      validateErrorInEvent(thenReturn);
      resolveCursorProviders(streamingHelper, thenReturn);
      mockModule.when(processor, withAttributes, thenReturn);
    }
  }

  /**
   * Verifies that the {@code processor} that has the given {@code withAttributes} has been executed.
   * <p>
   * During the test execution, if any processor is executed, and it matches the {@code processor} name and the
   * {@code withAttributes} it will be counted as called.
   * <p>
   * If the processor was not called the specified number of times, it will throw an {@link AssertionError}
   *
   * @param processor identifier of the processor to mock
   * @param withAttributes additional attributes to identify the processor
   * @param times exact number of times the processor should have been executed
   * @param atLeast minimum number of times the processor should have been executed
   * @param atMost maximum number of times the processor should have been executed
   */
  @Summary("Verify that a processor is called")
  public void verifyCall(@Example("mule:logger") String processor,
                         @Optional @Expression(NOT_SUPPORTED) List<Attribute> withAttributes,
                         @Optional Integer times,
                         @Optional Integer atLeast,
                         @Optional Integer atMost) {

    if (null == times && null == atLeast && null == atMost) {
      times = 1;
    }
    mockModule.setRegistry(registry);
    mockModule.verifyCall(processor, withAttributes, times, atLeast, atMost);
  }

  /**
   * Spies the {@code processor} that has the given {@code withAttributes}, executing logic before and after its execution.
   * <p>
   * During the test execution, if any processor tries to be executed, and it matches the {@code processor} name and the
   * {@code withAttributes}, the {@code beforeCall} will be executed before the processor, and the {@code afterCall} will be
   * executed after
   *
   * @param processor identifier of the processor to mock
   * @param withAttributes additional attributes to identify the processor
   * @param beforeCall processors to be executed before the spied processor
   * @param afterCall processors to be executed after the spied processor
   */
  @Summary("Allows to take actions over the event before and after the execution of a processor")
  public void spy(String processor,
                  @Optional @NullSafe @Expression(NOT_SUPPORTED) List<Attribute> withAttributes,
                  @Optional BeforeCall beforeCall,
                  @Optional AfterCall afterCall,
                  RouterCompletionCallback callback) {
    mockModule.setRegistry(registry);
    mockModule.spy(processor, withAttributes,
                   beforeCall == null ? null : beforeCall.getChain(),
                   afterCall == null ? null : afterCall.getChain());
    callback.success(Result.builder().build());
  }

  protected void setRegistry(Registry registry) {
    this.registry = registry;
  }

  protected void setMockModule(MockModule mockModule) {
    this.mockModule = mockModule;
  }

  private void validateErrorInEvent(Event thenReturn) {
    if (thenReturn != null && thenReturn.getError() != null) {
      if (StringUtils.isNotBlank(thenReturn.getError().getTypeId()) && thenReturn.getError().getCause() != null) {
        throw new IllegalArgumentException("Mocked event's error failure. The attributes typeId and cause are mutually exclusive. Please define only one of them");
      }
    }
  }
}
