/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.util.queue.internal;

import static java.util.Optional.ofNullable;

import java.io.Serializable;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import org.mule.munit.tools.util.queue.api.TemporaryQueueRule;
import org.mule.munit.tools.util.queue.internal.error.QueueErrorTypeProvider;
import org.mule.munit.tools.util.queue.internal.error.QueueTimeOutException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.runtime.operation.Result;

/**
 * Operations to store values on a temporary queue
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class QueueOperations {

  @Inject
  protected TemporaryQueueRule temporaryQueueRule;

  /**
   * Pushes the given value to a temporary queue so this can be consumed later on demand
   *
   * @param value Value to queue
   * @param queueName The name of the queue where dequeue a value
   */
  @Throws(QueueErrorTypeProvider.class)
  public void queue(@Content TypedValue<Serializable> value,
                    @Optional String queueName) {
    temporaryQueueRule.push(value, queueName);
  }

  /**
   * Pops the given data from a temporary queue.
   *
   * @param queueName The name of the queue where dequeue a value
   * @param timeout Time to wait for a value until the connector throws a QUEUE_TIMEOUT error
   * @param timeoutUnit Time Unit to use to wait for a value to arrive
   * @return The next queued value
   */
  @Throws(QueueErrorTypeProvider.class)
  public Result<Serializable, Void> dequeue(@Optional String queueName,
                                            @Optional(defaultValue = "10000") Long timeout,
                                            @Optional(defaultValue = "MILLISECONDS") TimeUnit timeoutUnit) {
    return ofNullable(temporaryQueueRule.pop(queueName, timeoutUnit.toMillis(timeout))).map(this::asResult)
        .orElseThrow(() -> queueName == null ? new QueueTimeOutException() : new QueueTimeOutException(queueName));
  }

  private Result<Serializable, Void> asResult(Object value) {
    TypedValue<Serializable> typedValue = value instanceof TypedValue
        ? (TypedValue<Serializable>) value
        : new TypedValue<>((Serializable) value, DataType.fromType(value.getClass()));

    return Result.<Serializable, Void>builder()
        .output(typedValue.getValue())
        .mediaType(typedValue.getDataType().getMediaType())
        .build();
  }
}
