/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.util;

import static java.nio.charset.StandardCharsets.ISO_8859_1;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.UnsupportedCharsetException;

import org.junit.Before;
import org.junit.Test;
import org.mule.munit.common.util.IOUtils;

public class GetResourceFunctionsTest {

  private static final String FILE_CONTENT = String.format("EXAMPLE_PAYLOAD%n");
  private static final String FILE_NAME = "util/files/example.txt";

  private GetResourceFunctions functions;

  @Before
  public void setUp() {
    functions = new GetResourceFunctions();
  }

  @Test
  public void asString() throws Exception {
    assertThat(functions.getResourceAsString(FILE_NAME, null), equalTo(FILE_CONTENT));
  }

  @Test
  public void asStringWithEncoding() throws Exception {
    String encodedString = IOUtils.toString(IOUtils.getResourceAsUrl("util/files/exampleISO.txt", getClass()), ISO_8859_1);
    assertThat(functions.getResourceAsString("util/files/exampleISO.txt", ISO_8859_1.name()), equalTo(encodedString));
  }

  @Test(expected = UnsupportedCharsetException.class)
  public void asStringWithNonExistentEncoding() throws Exception {
    functions.getResourceAsString(FILE_NAME, "NON_EXISTENT_ENCODING");
  }

  @Test
  public void asByteArray() throws Exception {
    byte[] value = functions.getResourceAsByteArray(FILE_NAME);
    assertThat(IOUtils.toString(value, Charset.defaultCharset().name()), equalTo(FILE_CONTENT));
  }

  @Test
  public void asStream() throws Exception {
    InputStream value = functions.getResourceAsStream(FILE_NAME);
    assertThat(IOUtils.toString(value, Charset.defaultCharset().name()), equalTo(FILE_CONTENT));
  }

  @Test
  public void asReusableStream() throws Exception {
    InputStream value = functions.getResourceAsReusableStream(FILE_NAME);
    assertThat(IOUtils.toString(value, Charset.defaultCharset().name()), equalTo(FILE_CONTENT));
    assertThat(IOUtils.toString(value, Charset.defaultCharset().name()), equalTo(FILE_CONTENT));
  }

}
