/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.assertion;


import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.munit.assertion.api.TypedValue.fromMuleTypedValue;
import static org.mule.runtime.api.meta.ExpressionSupport.REQUIRED;

import javax.inject.Inject;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.assertion.api.MunitAssertion;
import org.mule.munit.assertion.api.expression.MatcherResult;
import org.mule.munit.assertion.api.matchers.Matcher;
import org.mule.munit.assertion.internal.AssertModule;
import org.mule.munit.assertion.internal.HamcrestFactory;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.util.IOUtils;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.metadata.CollectionDataType;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.dsl.xml.ParameterDsl;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.ClassValue;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Example;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.parameter.ParameterResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>
 * Operations to perform assertions
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class AssertOperations {

  private static final String RUN_CUSTOM_INTERFACE_ERROR =
      "The run-custom operation requires a class implementing %s. Please provide one.";

  private static final String RUN_CUSTOM_ERROR = "Unable to create custom assertion";
  private static final String MUNIT_ASSERTION_CLASS = "org.mule.munit.assertion.api.MunitAssertion";
  private static final String DEFAULT_ASSERT_CONTENT = "#[import * from dw::test::Asserts \n ---\npayload must notBeNull()]";
  private static final String LOCATION_MESSAGE = " at file: [%s], line: [%d]";
  private static final String MUNIT_BINDING_NAME = "munit_equal_to_placeholder";
  private static final String EQUAL_TO_WRAPPER = "#[MunitTools::equalTo(" + MUNIT_BINDING_NAME + ")]";

  private static final DataType MATCHER_DATA_TYPE = DataType.fromType(Matcher.class);
  private static final Logger logger = LoggerFactory.getLogger(AssertOperations.class);

  private AssertModule assertModule = new AssertModule();

  @Inject
  public ExpressionManager expressionManager;

  /**
   * Checks if the {@code expression} matches the given {@code is} Matcher. If the assertion fails, it throws an
   * {@link AssertionError}
   *
   * @param expression value to be asserted
   * @param is matcher to validate against the expression
   * @param message message in case the assertion fails
   */
  @Summary("Perform an assertion over an expression")
  public void assertThat(@Example("#[payload]") TypedValue<Object> expression,
                         @Expression(REQUIRED) @Example("#[MunitTools::notNullValue()]") Matcher is,
                         @Optional(defaultValue = StringUtils.EMPTY) String message, ComponentLocation location) {
    assertModule.assertThat(message + locationMessage(location), validateExpression(expression), HamcrestFactory.create(is));
  }

  private TypedValue<Object> validateExpression(TypedValue<Object> expression) {
    if (!expression.getDataType().isStreamType() || expression.getDataType() instanceof CollectionDataType) {
      return expression;
    }

    try {
      byte[] bytes = IOUtils.toByteArray((CursorStream) expression.getValue());
      DataType dataType = DataType.builder(expression.getDataType()).type(bytes.getClass()).build();
      return new TypedValue<Object>(bytes, dataType, expression.getByteLength());
    } catch (Exception e) {
      logger.warn("Failed to convert 'expression' to byte[], returning the original value: ", e);
      return expression;
    }
  }

  /**
   * Evaluates if the given {@code that} assertion expression is successful. If the assertion fails, it throws an
   * {@link AssertionError}
   *
   * @param that result of the assertion expression
   * @param message message in case the assertion fails
   */
  @Summary("Asserts that an expression is successful")
  @DisplayName("Assert expression")
  @Alias("assert")
  @Throws(AssertionErrorProvider.class)
  public void assertExpression(@DisplayName("Expression") @Optional(
      defaultValue = DEFAULT_ASSERT_CONTENT) @Content ParameterResolver<MatcherResult> that,
                               @Optional(defaultValue = "Assertion failed") String message, ComponentLocation location) {
    try {
      assertModule.assertMatcherResult(that.resolve(), message + locationMessage(location));
    } catch (ExpressionRuntimeException e) {
      if (e.getMessage().contains(format("to class '%s'", MatcherResult.class.getSimpleName()))) {
        throw new InvalidAssertionExpressionException(that.getExpression().orElse(EMPTY), e);
      } else {
        throw e;
      }
    }
  }

  /**
   * Checks if the given {@code actual} value is equal to the {@code expected} value. If they differ, an {@link AssertionError} is
   * thrown.
   *
   * @param actual actual value to be asserted
   * @param expected expected value of the assertion
   * @param message message in case the assertion fails
   */
  @Summary("Check if an expression is equal to a value")
  public void assertEquals(@Example("#[payload]") TypedValue<Object> actual,
                           @ParameterDsl(allowReferences = false) @Example("#['EXAMPLE']") TypedValue<Object> expected,
                           @Optional(defaultValue = StringUtils.EMPTY) String message, ComponentLocation location) {
    BindingContext bindingContext = BindingContext.builder().addBinding(MUNIT_BINDING_NAME, expected).build();
    Matcher equalToMatcher = (Matcher) expressionManager.evaluate(EQUAL_TO_WRAPPER, MATCHER_DATA_TYPE, bindingContext).getValue();
    assertModule.assertThat(message + locationMessage(location), actual, HamcrestFactory.create(equalToMatcher));
  }

  /**
   * Fails with a {@link AssertionError} with the given {@code message}
   *
   * @param message message for the assertion failure
   */
  @Summary("Fail with an assertion")
  public void fail(@Optional(defaultValue = StringUtils.EMPTY) String message) {
    throw java.util.Optional.ofNullable(message).map(AssertionError::new).orElseGet(AssertionError::new);
  }

  /**
   * Runs a custom {@code assertion} over the given {@code expression} with the parameters {@code params}
   * <p>
   * The assertion must implement {@link MunitAssertion}
   *
   * @param assertion full qualified name of the custom assertion class
   * @param expression value to be asserted
   * @param params parameters of the custom assertion
   */
  @Summary("Run a custom assertion")
  public void runCustom(@Example("com.example.CustomAssertion") @ClassValue(
      extendsOrImplements = MUNIT_ASSERTION_CLASS) String assertion,
                        @Example("#[payload]") TypedValue<Object> expression,
                        @Optional Object params) {
    if (isBlank(assertion)) {
      throw new MunitError(format(RUN_CUSTOM_INTERFACE_ERROR, MUNIT_ASSERTION_CLASS));
    }
    try {
      MunitAssertion munitAssertion = (MunitAssertion) Class.forName(assertion).newInstance();
      munitAssertion.execute(fromMuleTypedValue(expression), params);
    } catch (ClassCastException e) {
      throw new MunitError(format(RUN_CUSTOM_INTERFACE_ERROR, MUNIT_ASSERTION_CLASS));
    } catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
      throw new MunitError(RUN_CUSTOM_ERROR, e);
    }
  }

  protected void setAssertModule(AssertModule assertModule) {
    this.assertModule = assertModule;
  }

  private String locationMessage(ComponentLocation location) {
    return format(LOCATION_MESSAGE, location.getFileName().orElse("UNKNOWN"), location.getLineInFile().orElse(-1));
  }

}
