/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package mock;

import org.junit.Rule;
import org.mule.munit.common.api.model.Answer;
import org.mule.munit.common.api.model.Attribute;
import org.mule.munit.common.api.model.Event;
import org.mule.munit.common.api.model.FlowName;
import org.mule.munit.mock.MockModule;
import org.mule.munit.tools.mock.AfterCall;
import org.mule.munit.tools.mock.BeforeCall;
import org.mule.munit.tools.mock.MockOperations;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.extension.api.runtime.process.RouterCompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.rules.ExpectedException;
import org.mockito.*;
import org.mockito.junit.MockitoJUnitRunner;

import java.lang.reflect.Field;
import java.util.Collections;
import java.util.List;

import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class MockOperationsTest {

  private MockOperations mockOperations;

  @Mock
  private Registry registry;

  @Mock
  private MockModule mockModule;

  @Mock
  private Answer answer;

  @Mock
  private StreamingHelper streamingHelper;

  @Mock
  private Event event;

  @Mock
  private FlowName flowName;

  @Mock
  private Attribute attribute;

  @Mock
  private RouterCompletionCallback callback;

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Before
  public void setUp() throws Exception {
    mockOperations = new MockOperations();
    setPrivateField(mockOperations, "registry", registry);
    setPrivateField(mockOperations, "mockModule", mockModule);
  }

  /**
   * Utility method to set private/protected fields using reflection.
   */
  private void setPrivateField(Object object, String fieldName, Object value) throws Exception {
    Field field = object.getClass().getDeclaredField(fieldName);
    field.setAccessible(true);
    field.set(object, value);
  }

  @Test
  public void testMockWhen_WithThenCall() {
    List<Attribute> attributes = Collections.singletonList(attribute);

    when(answer.getThenCall()).thenReturn(flowName);
    mockOperations.mockWhen("testProcessor", attributes, answer, streamingHelper);

    verify(mockModule).setRegistry(registry);
    verify(mockModule).when("testProcessor", attributes, flowName);
  }

  @Test
  public void testMockWhen_WithThenReturn() {
    List<Attribute> attributes = Collections.singletonList(attribute);

    when(answer.getThenReturn()).thenReturn(event);
    when(answer.getThenCall()).thenReturn(null);

    mockOperations.mockWhen("testProcessor", attributes, answer, streamingHelper);

    verify(mockModule).setRegistry(registry);
    verify(mockModule).when("testProcessor", attributes, event);
  }

  @Test
  public void testVerifyCall_WithTimes() {
    List<Attribute> attributes = Collections.singletonList(attribute);
    mockOperations.verifyCall("testProcessor", attributes, 2, null, null);

    verify(mockModule).setRegistry(registry);
    verify(mockModule).verifyCall("testProcessor", attributes, 2, null, null);
  }

  @Test
  public void testVerifyCall_WithAtLeastAtMost() {
    List<Attribute> attributes = Collections.singletonList(attribute);
    mockOperations.verifyCall("testProcessor", attributes, null, 1, 3);

    verify(mockModule).setRegistry(registry);
    verify(mockModule).verifyCall("testProcessor", attributes, null, 1, 3);
  }

  @Test
  public void testVerifyCall_DefaultTimes() {
    List<Attribute> attributes = Collections.singletonList(attribute);
    mockOperations.verifyCall("testProcessor", attributes, null, null, null);

    verify(mockModule).setRegistry(registry);
    verify(mockModule).verifyCall("testProcessor", attributes, 1, null, null);
  }

  @Test
  public void testVerifyCall_ExceptionThrown() {
    expectedException.expect(IllegalArgumentException.class);
    expectedException
        .expectMessage("VerifyCall error. The attribute times cannot be used in conjunction with the attributes atLeast or atMost");

    mockOperations.verifyCall("testProcessor", null, 2, 1, null);
  }

  @Test
  public void testSpy() {
    List<Attribute> attributes = Collections.singletonList(attribute);
    BeforeCall beforeCall = mock(BeforeCall.class);
    AfterCall afterCall = mock(AfterCall.class);

    when(beforeCall.getChain()).thenReturn(null);
    when(afterCall.getChain()).thenReturn(null);

    mockOperations.spy("testProcessor", attributes, beforeCall, afterCall, callback);

    verify(mockModule).setRegistry(registry);
    verify(mockModule).spy("testProcessor", attributes, null, null);
    verify(callback).success(any());
  }

  @Test
  public void testValidateErrorInEvent_NoError() {
    mockOperations.mockWhen("testProcessor", null, answer, streamingHelper);
    verify(mockModule).setRegistry(registry);
  }

}
