/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package mock;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;
import org.mule.munit.common.api.model.Event;
import org.mule.munit.common.api.model.EventAttributes;
import org.mule.munit.common.api.model.Payload;
import org.mule.munit.common.api.model.Variable;

import org.mule.munit.tools.mock.StreamingUtils;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.util.Arrays;
import java.util.Collections;
import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class StreamingUtilsTest {

  @Mock
  private StreamingHelper streamingHelper;

  @Mock
  private Payload payload;

  @Mock
  private EventAttributes eventAttributes;

  @Mock
  private Variable variable;

  @Mock
  private Event event;

  @Test
  public void testResolveCursorProviders_NullEvent() {
    // Ensure no exceptions are thrown when event is null
    StreamingUtils.resolveCursorProviders(streamingHelper, null);
  }

  @Test
  public void testResolveCursorProviders_WithValidEvent() {
    when(event.getPayload()).thenReturn(payload);
    when(event.getAttributes()).thenReturn(eventAttributes);
    when(event.getVariables()).thenReturn(Collections.singletonList(variable));

    when(payload.getValue()).thenReturn("payloadValue");
    when(eventAttributes.getValue()).thenReturn("attributeValue");
    when(variable.getValue()).thenReturn("variableValue");

    when(streamingHelper.resolveCursorProvider("payloadValue")).thenReturn("resolvedPayload");
    when(streamingHelper.resolveCursorProvider("attributeValue")).thenReturn("resolvedAttribute");
    when(streamingHelper.resolveCursorProvider("variableValue")).thenReturn("resolvedVariable");

    StreamingUtils.resolveCursorProviders(streamingHelper, event);

    verify(payload).setValue("resolvedPayload");
    verify(eventAttributes).setValue("resolvedAttribute");
    verify(variable).setValue("resolvedVariable");
  }

  @Test
  public void testResolveCursorProviders_WithNullPayload() {
    when(event.getPayload()).thenReturn(null);
    when(event.getAttributes()).thenReturn(eventAttributes);
    when(event.getVariables()).thenReturn(Collections.singletonList(variable));

    when(eventAttributes.getValue()).thenReturn("attributeValue");
    when(variable.getValue()).thenReturn("variableValue");

    when(streamingHelper.resolveCursorProvider("attributeValue")).thenReturn("resolvedAttribute");
    when(streamingHelper.resolveCursorProvider("variableValue")).thenReturn("resolvedVariable");

    StreamingUtils.resolveCursorProviders(streamingHelper, event);

    verify(eventAttributes).setValue("resolvedAttribute");
    verify(variable).setValue("resolvedVariable");
  }

  @Test
  public void testResolveCursorProviders_WithNullAttributes() {
    when(event.getPayload()).thenReturn(payload);
    when(event.getAttributes()).thenReturn(null);
    when(event.getVariables()).thenReturn(Collections.singletonList(variable));

    when(payload.getValue()).thenReturn("payloadValue");
    when(variable.getValue()).thenReturn("variableValue");

    when(streamingHelper.resolveCursorProvider("payloadValue")).thenReturn("resolvedPayload");
    when(streamingHelper.resolveCursorProvider("variableValue")).thenReturn("resolvedVariable");

    StreamingUtils.resolveCursorProviders(streamingHelper, event);

    verify(payload).setValue("resolvedPayload");
    verify(variable).setValue("resolvedVariable");
  }

  @Test
  public void testResolveCursorProviders_WithNullVariables() {
    when(event.getPayload()).thenReturn(payload);
    when(event.getAttributes()).thenReturn(eventAttributes);
    when(event.getVariables()).thenReturn(null);

    when(payload.getValue()).thenReturn("payloadValue");
    when(eventAttributes.getValue()).thenReturn("attributeValue");

    when(streamingHelper.resolveCursorProvider("payloadValue")).thenReturn("resolvedPayload");
    when(streamingHelper.resolveCursorProvider("attributeValue")).thenReturn("resolvedAttribute");

    StreamingUtils.resolveCursorProviders(streamingHelper, event);

    verify(payload).setValue("resolvedPayload");
    verify(eventAttributes).setValue("resolvedAttribute");
  }

  @Test
  public void testResolveCursorProviders_WithEmptyVariables() {
    when(event.getPayload()).thenReturn(payload);
    when(event.getAttributes()).thenReturn(eventAttributes);
    when(event.getVariables()).thenReturn(Collections.emptyList());

    when(payload.getValue()).thenReturn("payloadValue");
    when(eventAttributes.getValue()).thenReturn("attributeValue");

    when(streamingHelper.resolveCursorProvider("payloadValue")).thenReturn("resolvedPayload");
    when(streamingHelper.resolveCursorProvider("attributeValue")).thenReturn("resolvedAttribute");

    StreamingUtils.resolveCursorProviders(streamingHelper, event);

    verify(payload).setValue("resolvedPayload");
    verify(eventAttributes).setValue("resolvedAttribute");
  }

  @Test
  public void testResolveCursorProviders_WithMultipleVariables() {
    Variable variable1 = mock(Variable.class);
    Variable variable2 = mock(Variable.class);

    when(event.getPayload()).thenReturn(payload);
    when(event.getAttributes()).thenReturn(eventAttributes);
    when(event.getVariables()).thenReturn(Arrays.asList(variable1, variable2));

    when(payload.getValue()).thenReturn("payloadValue");
    when(eventAttributes.getValue()).thenReturn("attributeValue");
    when(variable1.getValue()).thenReturn("var1");
    when(variable2.getValue()).thenReturn("var2");

    when(streamingHelper.resolveCursorProvider("payloadValue")).thenReturn("resolvedPayload");
    when(streamingHelper.resolveCursorProvider("attributeValue")).thenReturn("resolvedAttribute");
    when(streamingHelper.resolveCursorProvider("var1")).thenReturn("resolvedVar1");
    when(streamingHelper.resolveCursorProvider("var2")).thenReturn("resolvedVar2");

    StreamingUtils.resolveCursorProviders(streamingHelper, event);

    verify(payload).setValue("resolvedPayload");
    verify(eventAttributes).setValue("resolvedAttribute");
    verify(variable1).setValue("resolvedVar1");
    verify(variable2).setValue("resolvedVar2");
  }

  @Test(expected = NullPointerException.class)
  public void testResolveCursorProvider_NullStreamingHelper() throws Exception {
    invokePrivateMethod("resolveCursorProvider", null, "testValue");
  }

  private Object invokePrivateMethod(String methodName, Object... args) throws Exception {
    Class<?>[] paramTypes = Arrays.stream(args)
        .map(Object::getClass)
        .toArray(Class<?>[]::new);

    java.lang.reflect.Method method = StreamingUtils.class.getDeclaredMethod(methodName, paramTypes);
    method.setAccessible(true);
    return method.invoke(null, args);
  }
}
