/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package util;

import static org.junit.Assert.*;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.file.Files;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.mule.munit.tools.util.GetResourceFunctions;

public class GetResourceFunctionsTest {

  private static final String TEST_RESOURCE_PATH = "test-resource.txt";
  private static final String TEST_CONTENT = "Mock Resource Content";
  private static final String TEST_ENCODING = "UTF-8";

  private static File testFile;

  @BeforeClass
  public static void setUp() throws Exception {
    // Create test resource file inside the classpath (src/test/resources)
    testFile = new File(GetResourceFunctionsTest.class.getClassLoader().getResource("").getPath(), TEST_RESOURCE_PATH);

    if (!testFile.exists()) {
      try (FileOutputStream fos = new FileOutputStream(testFile)) {
        fos.write(TEST_CONTENT.getBytes(TEST_ENCODING));
      }
    }
  }

  @AfterClass
  public static void tearDown() throws Exception {
    // Delete test file after test execution
    if (testFile.exists()) {
      Files.delete(testFile.toPath());
    }
  }

  @Test
  public void testGetResourceAsByteArray() {
    byte[] result = GetResourceFunctions.getResourceAsByteArray(TEST_RESOURCE_PATH);
    assertNotNull(result);
    assertEquals(TEST_CONTENT, new String(result, Charset.forName(TEST_ENCODING)));
  }

  @Test
  public void testGetResourceAsStringWithEncoding() {
    String result = GetResourceFunctions.getResourceAsString(TEST_RESOURCE_PATH, TEST_ENCODING);
    assertEquals(TEST_CONTENT, result);
  }

  @Test
  public void testGetResourceAsStringWithDefaultEncoding() {
    String result = GetResourceFunctions.getResourceAsString(TEST_RESOURCE_PATH, null);
    assertEquals(TEST_CONTENT, result);
  }

  @Test(expected = IllegalArgumentException.class)
  public void testGetResourceAsByteArrayWithInvalidPath() {
    GetResourceFunctions.getResourceAsByteArray("invalid-path.txt");
  }

  @Test
  public void testGetResourceAsStream() {
    InputStream result = GetResourceFunctions.getResourceAsStream(TEST_RESOURCE_PATH);
    assertNotNull(result);
  }

  @Test
  public void testGetResourceAsReusableStream() {
    InputStream result = GetResourceFunctions.getResourceAsReusableStream(TEST_RESOURCE_PATH);
    assertNotNull(result);
  }
}
