/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package util;


import static org.junit.Assert.*;

import org.junit.Test;
import org.mule.munit.tools.util.MuleComponentsFunctions;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.message.Message;

import java.lang.reflect.Method;

public class MuleComponentsFunctionsTest {

  @Test
  public void testCreateMessageWithAllParameters() {
    Object payload = "Test Payload";
    String mimeType = MediaType.APPLICATION_JSON.toString();
    Object attributes = "Test Attributes";
    String attributesMimeType = "text/plain"; // Directly using "text/plain"

    Message message = MuleComponentsFunctions.createMessage(payload, mimeType, attributes, attributesMimeType);

    assertNotNull(message);
    assertEquals(payload, message.getPayload().getValue());
    assertEquals(mimeType, message.getPayload().getDataType().getMediaType().toString());
    assertEquals(attributes, message.getAttributes().getValue());
    assertEquals(attributesMimeType, message.getAttributes().getDataType().getMediaType().toString());
  }

  @Test
  public void testCreateMessageWithNullMimeTypes() {
    Object payload = "Test Payload";
    Object attributes = "Test Attributes";

    Message message = MuleComponentsFunctions.createMessage(payload, null, attributes, null);

    assertNotNull(message);
    assertEquals(payload, message.getPayload().getValue());
    assertEquals(MediaType.ANY.toString(), message.getPayload().getDataType().getMediaType().toString()); // Expecting "*/*"
    assertEquals(attributes, message.getAttributes().getValue());
    assertEquals(MediaType.ANY.toString(), message.getAttributes().getDataType().getMediaType().toString()); // Expecting "*/*"
  }

  @Test
  public void testCreateMessageWithNullPayloadAndAttributes() {
    Message message = MuleComponentsFunctions.createMessage(null, null, null, null);

    assertNotNull(message);
    assertNull(message.getPayload().getValue());
    assertEquals(MediaType.ANY.toString(), message.getPayload().getDataType().getMediaType().toString()); // Expecting "*/*"
    assertNull(message.getAttributes().getValue());
    assertEquals(MediaType.ANY.toString(), message.getAttributes().getDataType().getMediaType().toString()); // Expecting "*/*"
  }

  @Test
  public void testPrivateMethodGetTypedValueWithMimeType() throws Exception {
    Method method = MuleComponentsFunctions.class.getDeclaredMethod("getTypedValue", Object.class, String.class);
    method.setAccessible(true);

    String testValue = "Test Value";
    String mimeType = MediaType.APPLICATION_XML.toString();
    TypedValue<?> result = (TypedValue<?>) method.invoke(null, testValue, mimeType);

    assertNotNull(result);
    assertEquals(testValue, result.getValue());
    assertEquals(mimeType, result.getDataType().getMediaType().toString());
  }

  @Test
  public void testPrivateMethodGetTypedValueWithNullMimeType() throws Exception {
    Method method = MuleComponentsFunctions.class.getDeclaredMethod("getTypedValue", Object.class, String.class);
    method.setAccessible(true);

    String testValue = "Test Value";
    TypedValue<?> result = (TypedValue<?>) method.invoke(null, testValue, null);

    assertNotNull(result);
    assertEquals(testValue, result.getValue());
    assertEquals(MediaType.ANY.toString(), result.getDataType().getMediaType().toString()); // Expecting "*/*"
  }

  @Test(expected = Exception.class)
  public void testPrivateMethodGetTypedValueWithInvalidInvocation() throws Exception {
    Method method = MuleComponentsFunctions.class.getDeclaredMethod("getTypedValue", Object.class, String.class);
    method.setAccessible(true);

    // This test checks if an exception is thrown when invoking with invalid parameters
    method.invoke(null, null, 123); // Invalid type for second parameter (should be String)
  }
}
