/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package util.queue.api;

import java.io.Serializable;
import org.mule.munit.tools.util.queue.api.TemporaryQueueRule;
import org.mule.munit.tools.util.queue.internal.error.QueueTimeOutException;
import org.mule.runtime.core.api.util.queue.Queue;
import org.mule.runtime.core.api.util.queue.QueueManager;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.junit.MockitoJUnitRunner;
import org.mule.runtime.core.api.util.queue.QueueSession;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class TemporaryQueueRuleTest {

  private static final String TEST_QUEUE = "TEST_QUEUE";

  @InjectMocks
  private TemporaryQueueRule temporaryQueueRule;

  @Mock
  private QueueManager queueManager;

  @Mock
  private QueueSession queueSession;

  @Mock
  private Queue queue;

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Before
  public void setUp() {
    MockitoAnnotations.initMocks(this);
    when(queueManager.getQueueSession()).thenReturn(queueSession);
    when(queueSession.getQueue(TEST_QUEUE)).thenReturn(queue);
    when(queueManager.getQueueSession().getQueue(TEST_QUEUE)).thenReturn(queue);
  }

  @Test
  public void testInitialise() throws InterruptedException {
    temporaryQueueRule.initialise();

    // Indirectly verify that the initialization happened correctly
    temporaryQueueRule.push("testData", TEST_QUEUE);
    verify(queue, times(1)).put("testData");
  }

  @Test
  public void testPush() throws Exception {
    Serializable testData = "testData";
    doNothing().when(queue).put(testData);

    temporaryQueueRule.push(testData, TEST_QUEUE);

    verify(queue, times(1)).put(testData);
  }

  @Test
  public void testPopWithoutTimeout() throws Exception {
    Serializable expectedData = "testData";
    when(queue.take()).thenReturn(expectedData);

    Serializable actualData = temporaryQueueRule.pop(TEST_QUEUE, null);

    assertEquals(expectedData, actualData);
  }

  @Test
  public void testPopWithTimeout() throws Exception {
    Serializable expectedData = "testData";
    when(queue.poll(1000L)).thenReturn(expectedData);

    Serializable actualData = temporaryQueueRule.pop(TEST_QUEUE, 1000L);

    assertEquals(expectedData, actualData);
  }

  @Test
  public void testSize() {
    when(queue.size()).thenReturn(5);

    // Simulate private field modification through public API
    temporaryQueueRule.initialise();
    temporaryQueueRule.push("testData", TEST_QUEUE);

    int size = TemporaryQueueRule.size(TEST_QUEUE);

    assertEquals(5, size);
  }

  @Test
  public void testReset() throws InterruptedException {
    temporaryQueueRule.push("testData", TEST_QUEUE);
    doNothing().when(queue).clear();

    temporaryQueueRule.reset();

    verify(queue, times(1)).clear();
  }

  @Test
  public void testApply() throws InterruptedException {
    temporaryQueueRule.push("testData", TEST_QUEUE);
    doNothing().when(queue).clear();

    temporaryQueueRule.apply(null);

    verify(queue, times(1)).clear();
  }

  @Test
  public void testExecuteOnQueueThrowsQueueTimeOutException() throws Exception {
    when(queue.take()).thenThrow(new InterruptedException());

    expectedException.expect(QueueTimeOutException.class);
    expectedException.expectMessage(TEST_QUEUE);

    temporaryQueueRule.pop(TEST_QUEUE, null);
  }

}
