/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package util.queue.internal;

import java.io.Serializable;
import java.util.concurrent.TimeUnit;

import org.mule.munit.tools.util.queue.api.TemporaryQueueRule;
import org.mule.munit.tools.util.queue.internal.error.QueueTimeOutException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.operation.Result;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.lang.reflect.Method;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.junit.MockitoJUnitRunner;

import org.mule.munit.tools.util.queue.internal.QueueOperations;

@RunWith(MockitoJUnitRunner.class)
public class QueueOperationsTest {

  @Mock
  private TemporaryQueueRule temporaryQueueRule;

  @InjectMocks
  private QueueOperations queueOperations;

  @Before
  public void setUp() {
    MockitoAnnotations.initMocks(this);
  }

  @Test
  public void testQueue_Success() {
    TypedValue<Serializable> value = new TypedValue<>("Test Value", DataType.STRING);
    String queueName = "testQueue";

    queueOperations.queue(value, queueName);

    verify(temporaryQueueRule, times(1)).push(value, queueName);
  }

  @Test
  public void testDequeue_Success() {
    String queueName = "testQueue";
    Long timeout = 5000L;
    TimeUnit timeUnit = TimeUnit.MILLISECONDS;
    Serializable mockValue = "DequeuedValue";

    when(temporaryQueueRule.pop(queueName, timeUnit.toMillis(timeout))).thenReturn(mockValue);

    Result<Serializable, Void> result = queueOperations.dequeue(queueName, timeout, timeUnit);

    assertNotNull(result);
    assertEquals(mockValue, result.getOutput());
  }

  @Test(expected = QueueTimeOutException.class)
  public void testDequeue_ThrowsQueueTimeoutException_NoQueueName() {
    Long timeout = 5000L;
    TimeUnit timeUnit = TimeUnit.MILLISECONDS;

    when(temporaryQueueRule.pop(null, timeUnit.toMillis(timeout))).thenReturn(null);

    queueOperations.dequeue(null, timeout, timeUnit);
  }

  @Test(expected = QueueTimeOutException.class)
  public void testDequeue_ThrowsQueueTimeoutException_WithQueueName() {
    String queueName = "testQueue";
    Long timeout = 5000L;
    TimeUnit timeUnit = TimeUnit.MILLISECONDS;

    when(temporaryQueueRule.pop(queueName, timeUnit.toMillis(timeout))).thenReturn(null);

    queueOperations.dequeue(queueName, timeout, timeUnit);
  }

  @Test
  public void testAsResult_WithTypedValue() throws Exception {
    TypedValue<Serializable> typedValue = new TypedValue<>("Test", DataType.STRING);

    Method method = QueueOperations.class.getDeclaredMethod("asResult", Object.class);
    method.setAccessible(true);

    @SuppressWarnings("unchecked")
    Result<Serializable, Void> result = (Result<Serializable, Void>) method.invoke(queueOperations, typedValue);

    assertNotNull(result);
    assertEquals("Test", result.getOutput());
  }

  @Test
  public void testAsResult_WithRawValue() throws Exception {
    String rawValue = "RawValue";

    Method method = QueueOperations.class.getDeclaredMethod("asResult", Object.class);
    method.setAccessible(true);

    @SuppressWarnings("unchecked")
    Result<Serializable, Void> result = (Result<Serializable, Void>) method.invoke(queueOperations, rawValue);

    assertNotNull(result);
    assertEquals(rawValue, result.getOutput());
  }
}
