/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package util.store;

import org.mule.munit.runner.component.rules.TemporaryStorageRule;
import org.mule.munit.tools.util.store.KeyAlreadyExistsException;
import org.mule.munit.tools.util.store.MissingKeyException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.Optional;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mule.munit.tools.util.store.StorageOperations;
import org.mockito.junit.MockitoJUnitRunner;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

@RunWith(MockitoJUnitRunner.class)
public class StorageOperationsTest {

  @Mock
  private TemporaryStorageRule storageRule;

  @Mock
  private ObjectStore objectStore;

  @InjectMocks
  private StorageOperations storageOperations;

  @Before
  public void setUp() throws Exception {
    storageOperations = new StorageOperations();

    // Use reflection to set the protected storageRule field
    Field storageRuleField = StorageOperations.class.getDeclaredField("storageRule");
    storageRuleField.setAccessible(true);
    storageRuleField.set(storageOperations, storageRule);
  }

  @Test(expected = KeyAlreadyExistsException.class)
  public void testStore_KeyAlreadyExists() {
    String key = "existingKey";
    TypedValue<Serializable> value = new TypedValue<>("testValue", DataType.STRING);

    when(storageRule.retrieve(key)).thenReturn(java.util.Optional.of(value));

    storageOperations.store(key, value, true);
  }

  @Test
  public void testRetrieve_Success() {
    String key = "testKey";
    TypedValue<Serializable> value = new TypedValue<>("testValue", DataType.STRING);

    when(storageRule.retrieve(key)).thenReturn(Optional.of(value));
    Result<Serializable, Void> result = storageOperations.retrieve(key);

    assertEquals("testValue", result.getOutput());
  }

  @Test(expected = MissingKeyException.class)
  public void testRetrieve_KeyNotFound() {
    String key = "missingKey";
    when(storageRule.retrieve(key)).thenReturn(Optional.empty());
    storageOperations.retrieve(key);
  }

  @Test
  public void testRemove_Success() {
    String key = "testKey";
    TypedValue<Serializable> value = new TypedValue<>("testValue", DataType.STRING);

    when(storageRule.remove(key)).thenReturn(Optional.of(value));
    Result<Serializable, Void> result = storageOperations.remove(key);

    assertEquals("testValue", result.getOutput());
  }

  @Test(expected = MissingKeyException.class)
  public void testRemove_KeyNotFound() {
    String key = "missingKey";
    when(storageRule.remove(key)).thenReturn(Optional.empty());
    storageOperations.remove(key);
  }

  @Test
  public void testClearStoredData() {
    doNothing().when(storageRule).clear();
    storageOperations.clearStoredData();
    verify(storageRule, times(1)).clear();
  }

  @Test(expected = ModuleException.class)
  public void testValidateKey_NullOrEmpty() throws Exception {
    storageOperations.store("", new TypedValue<>("testValue", DataType.STRING), false);
  }

  @Test
  public void testStoreOAuthToken_Success() throws Exception {
    String accessToken = "accessToken";
    String ownerConfigName = "config";
    String resourceOwnerId = "ownerId";
    String key = resourceOwnerId + "-" + ownerConfigName;

    when(objectStore.contains(key)).thenReturn(false);
    doNothing().when(objectStore).store(eq(key), any());

    storageOperations.storeOAuthToken(accessToken, objectStore, ownerConfigName, resourceOwnerId, null, null, null, null, true);
    verify(objectStore).store(eq(key), any());
  }

  @Test(expected = ModuleException.class)
  public void testStoreOAuthToken_KeyAlreadyExists() throws Exception {
    String accessToken = "accessToken";
    String ownerConfigName = "config";
    String resourceOwnerId = "ownerId";
    String key = resourceOwnerId + "-" + ownerConfigName;

    when(objectStore.contains(key)).thenReturn(true);
    storageOperations.storeOAuthToken(accessToken, objectStore, ownerConfigName, resourceOwnerId, null, null, null, null, false);
  }
}
