/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package assertion;

import com.google.common.collect.ImmutableMap;
import org.mockito.ArgumentCaptor;
import org.mule.munit.assertion.api.expression.MatcherResult;
import org.mule.munit.assertion.api.matchers.Matcher;
import org.mule.munit.assertion.internal.AssertModule;
import org.mule.munit.assertion.internal.HamcrestFactory;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.tools.assertion.InvalidAssertionExpressionException;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.i18n.I18nMessage;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.extension.api.runtime.parameter.ParameterResolver;

import org.mule.munit.tools.assertion.AssertOperations;
import org.mule.runtime.core.internal.streaming.bytes.ByteArrayCursorStream;
import org.mule.runtime.core.internal.streaming.bytes.ByteArrayCursorStreamProvider;

import static org.mockito.Mockito.*;
import static org.junit.Assert.*;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.MockitoAnnotations;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;


public class AssertOperationsTest {

  @InjectMocks
  private AssertOperations assertOperations;

  @Mock
  private AssertModule assertModule;

  @Mock
  private ComponentLocation location;

  @Mock
  private ParameterResolver<Map<String, Object>> parameterResolver;

  private Map<String, Object> matcherResult = ImmutableMap.<String, Object>builder()
      .put("matches", true)
      .put("reasons", Collections.emptyList())
      .put("description", ImmutableMap.<String, Object>builder()
          .put("expected", "expected message")
          .put("actual", "actual message")
          .build())
      .build();

  @Mock
  private AssertModule mockAssertModule;

  @Mock
  private ExpressionManager mockExpressionManager;

  @Mock
  private ComponentLocation mockLocation;

  @Before
  public void setUp() throws Exception {
    MockitoAnnotations.initMocks(this);
    injectPrivateField(assertOperations, "assertModule", mockAssertModule);
    injectPrivateField(assertOperations, "expressionManager", mockExpressionManager);
  }

  private void injectPrivateField(Object target, String fieldName, Object value) throws Exception {
    Field field = target.getClass().getDeclaredField(fieldName);
    field.setAccessible(true);
    field.set(target, value);
  }

  @Test(expected = AssertionError.class)
  public void testFail_ThrowsAssertionError() {
    assertOperations.fail("Test Failure Message");
  }

  @Test(expected = MunitError.class)
  public void testRunCustom_MissingClass_ThrowsException() {
    assertOperations.runCustom("", new TypedValue<>("testValue", null), null);
  }

  @Test
  public void testLocationMessage_GeneratesCorrectMessage() throws Exception {
    when(mockLocation.getFileName()).thenReturn(Optional.of("testFile.xml"));
    when(mockLocation.getLineInFile()).thenReturn(Optional.of(42));

    Method method = AssertOperations.class.getDeclaredMethod("locationMessage", ComponentLocation.class);
    method.setAccessible(true);

    String result = (String) method.invoke(assertOperations, mockLocation);
    assertEquals(" at file: [testFile.xml], line: [42]", result);
  }

  @Test
  public void testAssertThat() {
    TypedValue<Object> expression = new TypedValue<>("testPayload", DataType.STRING);
    Matcher matcher = mock(Matcher.class);

    when(location.getFileName()).thenReturn(java.util.Optional.of("testFile.xml"));
    when(location.getLineInFile()).thenReturn(java.util.Optional.of(10));

    try (MockedStatic<HamcrestFactory> hamcrestFactoryMock = mockStatic(HamcrestFactory.class)) {
      org.hamcrest.Matcher hamcrestMatcher = mock(org.hamcrest.Matcher.class);
      hamcrestFactoryMock.when(() -> HamcrestFactory.create(matcher)).thenReturn(hamcrestMatcher);

      assertOperations.assertThat(expression, matcher, "Test Message", location);

      verify(mockAssertModule).assertThat(eq("Test Message at file: [testFile.xml], line: [10]"), any(), any());
    }
  }

  @Test
  public void testValidateExpressionExceptionHandling() throws Exception {
    TypedValue<Object> expression = mock(TypedValue.class);
    when(expression.getDataType()).thenReturn(DataType.fromType(CursorStream.class));
    when(expression.getValue()).thenThrow(new RuntimeException("Stream conversion failed"));

    Method validateExpressionMethod = AssertOperations.class.getDeclaredMethod("validateExpression", TypedValue.class);
    validateExpressionMethod.setAccessible(true);

    TypedValue<Object> result = (TypedValue<Object>) validateExpressionMethod.invoke(assertOperations, expression);
    assertEquals("Should return the original expression when an exception occurs", expression, result);
  }

  @Test
  public void testValidateExpressionFullTryBlockExecution() throws Exception {
    byte[] testData = "testPayload".getBytes(StandardCharsets.UTF_8);

    CursorStream cursorStream = new ByteArrayCursorStream(new ByteArrayCursorStreamProvider(testData), testData);
    TypedValue<Object> expression = mock(TypedValue.class);

    when(expression.getDataType()).thenReturn(DataType.fromType(CursorStream.class));
    when(expression.getValue()).thenReturn(cursorStream);
    when(expression.getByteLength()).thenReturn(OptionalLong.of(testData.length));

    Method validateExpressionMethod = AssertOperations.class.getDeclaredMethod("validateExpression", TypedValue.class);
    validateExpressionMethod.setAccessible(true);

    TypedValue<Object> result = (TypedValue<Object>) validateExpressionMethod.invoke(assertOperations, expression);
    assertArrayEquals("Should successfully convert stream to byte array", testData, (byte[]) result.getValue());
  }

  @Test
  public void testAssertExpressionSuccess() {
    ArgumentCaptor<MatcherResult> captor = ArgumentCaptor.forClass(MatcherResult.class);
    when(parameterResolver.resolve()).thenReturn(matcherResult);
    when(location.getFileName()).thenReturn(java.util.Optional.of("testFile.xml"));
    when(location.getLineInFile()).thenReturn(java.util.Optional.of(10));
    doCallRealMethod().when(mockAssertModule).assertMatcherResult(captor.capture(), anyString());

    assertOperations.assertExpression(parameterResolver, "Test Message", location);
    verify(mockAssertModule).assertMatcherResult(eq(captor.getValue()), eq("Test Message at file: [testFile.xml], line: [10]"));
  }

  @Test
  public void testAssertExpressionThrowsInvalidAssertionExpressionException() {
    I18nMessage mockMessage = mock(I18nMessage.class);
    when(mockMessage.getMessage()).thenReturn("to class 'MatcherResult'");

    ExpressionRuntimeException exceptionMock = mock(ExpressionRuntimeException.class);
    when(exceptionMock.getMessage()).thenReturn("to class 'MatcherResult'");
    when(exceptionMock.getCause()).thenReturn(new Exception());

    when(parameterResolver.resolve()).thenThrow(exceptionMock);
    when(parameterResolver.getExpression()).thenReturn(java.util.Optional.of("Invalid Expression"));

    try {
      assertOperations.assertExpression(parameterResolver, "Test Message", location);
      fail("Expected InvalidAssertionExpressionException was not thrown");
    } catch (InvalidAssertionExpressionException e) {
      assertTrue(e.getMessage().contains("Invalid Expression"));
    }
  }

  @Test
  public void testAssertExpressionThrowsExpressionRuntimeException() {
    ExpressionRuntimeException exceptionMock = mock(ExpressionRuntimeException.class);
    when(exceptionMock.getMessage()).thenReturn("Some other error");
    when(exceptionMock.getCause()).thenReturn(new Exception());

    when(parameterResolver.resolve()).thenThrow(exceptionMock);

    try {
      assertOperations.assertExpression(parameterResolver, "Test Message", location);
      fail("Expected ExpressionRuntimeException was not thrown");
    } catch (ExpressionRuntimeException e) {
      assertTrue(e.getMessage().contains("Some other error"));
    }
  }

}
