/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.assertion;

import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static org.hamcrest.CoreMatchers.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.util.List;
import java.util.function.Function;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mule.munit.assertion.api.matchers.Diff;
import org.mule.munit.assertion.api.matchers.Matcher;
import org.mule.munit.assertion.internal.AssertModule;
import org.mule.munit.common.exception.MunitError;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;

import com.example.StartsWithAssertion;

public class AssertOperationsTest {

  private AssertOperations assertOperations;

  @Before
  public void setUp() {
    assertOperations = new AssertOperations();
  }

  @Test
  public void assertThat() {
    AssertModule assertModuleMock = mock(AssertModule.class);
    assertOperations.setAssertModule(assertModuleMock);

    Object value = "";
    Diff.Difference difference = new Diff.Difference();
    difference.setExpected("Empty String");
    difference.setActual("was not empty");

    Diff diff = new Diff(false, singletonList(difference));

    TypedValue<Object> expression = new TypedValue<>(value, DataType.fromObject(value));
    Matcher is = new Matcher();
    is.setType("equalTo");
    is.setExpected(new Object[] {(Function<List<Object>, Diff>) objects -> diff});
    String message = "fail message 1";

    ComponentLocation locationMock = mock(ComponentLocation.class);
    when(locationMock.getLineInFile()).thenReturn(empty());
    when(locationMock.getFileName()).thenReturn(empty());
    String locationMessage = " at file: [UNKNOWN], line: [-1]";

    assertOperations.assertThat(expression, is, message, locationMock);
    verify(assertModuleMock, times(1)).assertThat(eq(message + locationMessage), eq(expression), any(org.hamcrest.Matcher.class));
  }

  @Test
  public void fail() {
    String message = "fail message 1";
    try {
      assertOperations.fail(message);
    } catch (AssertionError e) {
      Assert.assertThat(e.getMessage(), is(message));
    }
  }

  @Test(expected = MunitError.class)
  public void runCustomNullAssertion() {
    TypedValue<Object> expression = new TypedValue<>("", DataType.JSON_STRING);
    assertOperations.runCustom(null, expression, null);
  }

  @Test(expected = MunitError.class)
  public void runCustomNotMunitAssertion() {
    TypedValue<Object> expression = new TypedValue<>("", DataType.JSON_STRING);
    assertOperations.runCustom(String.class.getCanonicalName(), expression, null);
  }

  @Test(expected = AssertionError.class)
  public void runCustomAssertionFail() {
    TypedValue<Object> expression = new TypedValue<>("EXAMPLE", DataType.JSON_STRING);

    assertOperations.runCustom(StartsWithAssertion.class.getCanonicalName(), expression, "ZZ");
  }

  @Test
  public void runCustom() {
    TypedValue<Object> expression = new TypedValue<>("EXAMPLE", DataType.JSON_STRING);

    assertOperations.runCustom(StartsWithAssertion.class.getCanonicalName(), expression, "EX");
  }

}
