/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package util;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.charset.Charset;

import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.mule.munit.tools.util.MunitResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MunitResourceTest {

  private MunitResource munitResource;
  private ClassLoader mockClassLoader;
  private InputStream mockInputStream;

  @Before
  public void setUp() {
    mockClassLoader = mock(ClassLoader.class);
    mockInputStream = new ByteArrayInputStream("Test Data".getBytes(StandardCharsets.UTF_8));
    munitResource = new MunitResource("valid/path", mockClassLoader);
  }

  @Test
  public void testAsStream_Success() {
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(mockInputStream);
    InputStream result = munitResource.asStream();
    assertNotNull(result);
  }

  @Test(expected = IllegalArgumentException.class)
  public void testAsStream_ThrowsExceptionForInvalidPath() {
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(null);
    munitResource.asStream(); // Should throw IllegalArgumentException
  }

  @Test
  public void testAsString_Success() throws IOException {
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(mockInputStream);
    String result = munitResource.asString(StandardCharsets.UTF_8);
    assertEquals("Test Data", result);
  }

  @Test
  public void testAsString_ReturnsNullOnIOException() throws IOException {
    InputStream mockInputStream = mock(InputStream.class);
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(mockInputStream);
    doThrow(new IOException()).when(mockInputStream).read(any(byte[].class));

    String result = munitResource.asString(StandardCharsets.UTF_8);
    assertNull(result);
  }

  @Test
  public void testAsByteArray_Success() throws IOException {
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(mockInputStream);
    byte[] result = munitResource.asByteArray();
    assertNotNull(result);
    assertEquals("Test Data", new String(result, StandardCharsets.UTF_8));
  }

  @Test
  public void testAsByteArray_ReturnsNullOnIOException() throws IOException {
    InputStream mockInputStream = mock(InputStream.class);
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(mockInputStream);
    doThrow(new IOException()).when(mockInputStream).read(any(byte[].class));

    byte[] result = munitResource.asByteArray();
    assertNull(result);
  }

  @Test
  public void testAsReusableStream_Success() throws IOException {
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(mockInputStream);
    InputStream result = munitResource.asReusableStream();
    assertNotNull(result);
    assertEquals("Test Data", IOUtils.toString(result, StandardCharsets.UTF_8));
  }

  @Test
  public void testAsReusableStream_HandlesIOException() throws IOException {
    InputStream mockInputStream = mock(InputStream.class);
    when(mockClassLoader.getResourceAsStream("valid/path")).thenReturn(mockInputStream);
    doThrow(new IOException()).when(mockInputStream).read(any(byte[].class));

    InputStream result = munitResource.asReusableStream();
    assertNotNull(result);
    assertEquals("", IOUtils.toString(result, StandardCharsets.UTF_8)); // Empty content in case of failure
  }
}
